package legdb

import (
	"context"
	"time"

	pgx "github.com/jackc/pgx/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects/model"
)

// Guarantees order by carrier, flight number, operating_from, operating_to, operating_on_days, leg_seq_number
func (l *FlightLeg) Flights(dest chan *model.FlightPatternRecord, from time.Time, to time.Time) error {
	defer close(dest)
	sql := `SELECT
	fp.marketing_carrier as marketing_carrier,
	fp.marketing_flight_number as marketing_flight_number,
	fp.operating_from as operating_from,
	fp.operating_until as operating_until,
	fp.operating_on_days as operating_on_days,
	fp.arrival_day_shift as arrival_day_shift,
	fb.leg_seq_number as leg_seq_number,
	fb.departure_station as departure_station,
	fb.arrival_station as arrival_station
FROM flight_pattern fp
	INNER JOIN flight_base fb on fp.flight_base_id = fb.id
WHERE
	fp.operating_from <= $1
	AND fp.operating_until >= $2

UNION SELECT
	fp.marketing_carrier as marketing_carrier,
	fp.marketing_flight_number as marketing_flight_number,
	fp.operating_from as operating_from,
	fp.operating_until as operating_until,
	fp.operating_on_days as operating_on_days,
	fp.arrival_day_shift as arrival_day_shift,
	fb.leg_seq_number as leg_seq_number,
	fb.departure_station as departure_station,
	fb.arrival_station as arrival_station
FROM sirena_flight_pattern fp
	INNER JOIN sirena_flight_base fb on fp.flight_base_id = fb.id
WHERE
	fp.operating_from <= $1
	AND fp.operating_until >= $2

UNION SELECT
	fp.marketing_carrier as marketing_carrier,
	fp.marketing_flight_number as marketing_flight_number,
	fp.operating_from as operating_from,
	fp.operating_until as operating_until,
	fp.operating_on_days as operating_on_days,
	fp.arrival_day_shift as arrival_day_shift,
	fb.leg_seq_number as leg_seq_number,
	fb.departure_station as departure_station,
	fb.arrival_station as arrival_station
FROM apm_flight_pattern fp
	INNER JOIN apm_flight_base fb on fp.flight_base_id = fb.id
WHERE
	fp.operating_from <= $1
	AND fp.operating_until >= $2

ORDER BY
	marketing_carrier,
	marketing_flight_number,
	operating_from,
	operating_until,
	operating_on_days,
	leg_seq_number;`

	pool := l.cluster.RO()
	ctx, cancel := context.WithTimeout(context.Background(), l.bulkTimeout)
	defer cancel()
	conn, err := pool.Acquire(ctx)
	if err != nil {
		return xerrors.Errorf("error getting connection from pool: %w", err)
	}
	defer conn.Release()

	var tx pgx.Tx
	if tx, err = conn.Begin(ctx); err != nil {
		logger.Logger().Error("cannot begin transaction", log.Error(err))
		return err
	}
	defer tx.Rollback(ctx)

	var rows pgx.Rows
	rows, err = tx.Query(ctx, sql, to, from)
	if err != nil {
		if err == pgx.ErrNoRows {
			return nil
		} else {
			return xerrors.Errorf("error while scanning for leg number in database: %w", err)
		}
	}
	defer rows.Close()
	for rows.Next() {
		v := new(model.FlightPatternRecord)
		err = rows.Scan(
			&v.CarrierID,
			&v.Number,
			&v.OperatingFrom,
			&v.OperatingUntil,
			&v.OperatingOnDays,
			&v.ArrivalDayShift,
			&v.LegNumber,
			&v.DepartureStation,
			&v.ArrivalStation,
		)
		if err != nil {
			return xerrors.Errorf("error scanning row of flight: %w", err)
		}
		dest <- v
	}
	return nil
}
