package settings

import (
	"context"
	"time"

	"github.com/heetch/confita"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type Config struct {
	Parameters
	PG
	Server
	Logging
	YtLogging
	Logbroker
}

func (c *Config) LoadFromEnv() Config {
	loader := confita.NewLoader()
	logger.Logger().Info("Load config from environment's variables")

	appConfig := Config{
		Parameters{
			CacheSize:           2000000,
			CacheUpdateInterval: 15 * time.Minute,
			PrecacheWindow:      48 * time.Hour,
			UpdateStatusWorkers: 10,
		},
		PG{
			Hosts:     "localhost",
			User:      "root",
			Port:      5432,
			DataBase:  "avia_flight_board",
			Password:  "",
			Logging:   false,
			CurrentDC: "127.0.0.1",
		},
		Server{ListenPort: 8000},
		Logging{
			LogLevel:       log.InfoString,
			AppLogFileName: "status-importer.log",
		},
		YtLogging{
			UpdateStatusLogName:     "update-status.log",
			UpdateStatusLogDisabled: false,
		},
		Logbroker{
			ReadEndpoints: []string{
				"sas.logbroker.yandex.net",
				"vla.logbroker.yandex.net",
				"iva.logbroker.yandex.net",
			},
			StatusUpdater: LogbrokerReadableTopic{
				Topic: "/avia/development/flight-status/status",
			},
		},
	}
	err := loader.Load(context.Background(), &appConfig)
	if err != nil {
		logger.Logger().Fatal("Cannot load app configuration", log.Error(err))
	}

	if !appConfig.Logbroker.StatusUpdater.Disable {
		if len(appConfig.Logbroker.Token) == 0 {
			logger.Logger().Fatal("Logbroker token is not configured")
		}
		if len(appConfig.Logbroker.StatusUpdater.Topic) == 0 {
			logger.Logger().Fatal("Logbroker status updater topic is not configured")
		}
		if len(appConfig.Logbroker.StatusUpdater.Consumer) == 0 {
			logger.Logger().Fatal("Logbroker status updater consumer is not configured")
		}
	}

	if !appConfig.YtLogging.UpdateStatusLogDisabled {
		if len(appConfig.YtLogging.YtLogPath) == 0 {
			logger.Logger().Fatal("Yt log path is not configured")
		}
		if len(appConfig.YtLogging.UpdateStatusLogName) == 0 {
			logger.Logger().Fatal("Update status log name is not configured")
		}
	}

	return appConfig
}
