package status

import (
	"time"

	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
	appMetrics "a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/metrics"
	"a.yandex-team.ru/travel/library/go/metrics"
)

func labelsForStatus(status *tStatus) map[string]string {
	return map[string]string{
		"source":  status.Source,
		"airport": strutil.EncodeIcao9303(status.Airport),
	}
}

func invalidStatusMetrics(invalidStatuses Statuses) {
	for _, status := range invalidStatuses {
		metrics.GlobalAppMetrics().
			GetOrCreateCounter(
				appMetrics.StatusesValidation,
				labelsForStatus(status),
				appMetrics.InvalidRate,
			).Inc()
	}
}

func abnormalStatusMetrics(normalizeErrors tInvalidStatuses) {
	for _, status := range normalizeErrors {
		metrics.GlobalAppMetrics().
			GetOrCreateCounter(
				appMetrics.StatusesValidation,
				labelsForStatus(status.Status),
				appMetrics.AbnormalRate,
			).Inc()
	}
}

func failedStatusWriteMetrics(duration time.Duration, statuses Statuses) {
	statusWriteMetrics(duration, statuses, appMetrics.FailureTimings, appMetrics.FailureRate)
}

func successStatusWriteMetrics(duration time.Duration, statuses Statuses) {
	statusWriteMetrics(duration, statuses, appMetrics.SuccessTimings, appMetrics.SuccessRate)
}

func ignoredStatusWriteMetrics(duration time.Duration, statuses Statuses) {
	statusWriteMetrics(duration, statuses, appMetrics.IgnoredTimings, appMetrics.IgnoredRate)
}

func statusWriteMetrics(duration time.Duration, statuses Statuses, timingsName, rateName string) {
	now := time.Now()

	metrics.GlobalAppMetrics().
		GetOrCreateHistogram(
			appMetrics.StatusesWrite,
			nil,
			timingsName,
			appMetrics.StatusWriteTimingsBuckets,
		).RecordDuration(duration)

	for _, status := range statuses {
		metrics.GlobalAppMetrics().
			GetOrCreateCounter(
				appMetrics.StatusesWrite,
				labelsForStatus(status),
				rateName,
			).Inc()

		metrics.GlobalAppMetrics().
			GetOrCreateHistogram(
				appMetrics.StatusesDelay,
				nil,
				timingsName,
				appMetrics.StatusDelayTimingsBuckets,
			).RecordDuration(now.Sub(time.Unix(status.ReceivedAt, 0)))
	}
}
