package status

import (
	"strings"
	"time"
	"unicode"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/pkg/logging/yt/updatestatuslog"
	"a.yandex-team.ru/travel/library/go/errutil"
	"a.yandex-team.ru/travel/proto/avia/flight_status"
)

type tStatus struct {
	flight_status.FlightStatus
}

type StatusesToParse []tStatus

type Statuses []*tStatus

type ProcessingUnit struct {
	Statuses
	Finished chan error
}

func (ss Statuses) InitReceivedAt() {
	now := time.Now().UTC()
	nowUnix := now.Unix()
	for i := range ss {
		if ss[i].ReceivedAt == 0 {
			ss[i].ReceivedAt = nowUnix
		}
	}
}

type tInvalidStatus struct {
	Status *tStatus
	Err    string
}

type tInvalidStatuses []tInvalidStatus

func (is tInvalidStatuses) Statuses() Statuses {
	var statuses Statuses
	for _, s := range is {
		statuses = append(statuses, s.Status)
	}
	return statuses
}

func onlyDigits(s string) bool {
	for _, r := range s {
		if !unicode.IsDigit(r) {
			return false
		}
	}
	return true
}

func (status *tStatus) Validate() bool {
	if _, err := direction.FromString(status.Direction); err != nil {
		return false
	}
	return status.Status != "" &&
		len(status.AirlineCode) > 0 &&
		status.FlightNumber != "" &&
		status.FlightDate != "" && !onlyDigits(status.AirlineCode) &&
		!fromDistantPast(status.FlightDate)
}

func fromDistantPast(date string) bool {
	t, err := time.Parse("2006-01-02", date)
	if err != nil {
		return false
	}
	if time.Since(t) > 7*24*time.Hour {
		return true
	}
	return false
}

func (status *tStatus) Normalize(objects *objects.Objects) (err error) {
	defer errutil.Wrap(&err, "tStatus.Normalize(%v)", status)
	scheduledTimeNorm, err := normalizeTimeString(status.TimeScheduled)
	if err != nil {
		return err
	}
	actualTimeNorm, err := normalizeTimeString(status.TimeActual)
	if err != nil {
		return err
	}

	carriers := objects.Carrier.ByCode(status.AirlineCode)
	if len(carriers) != 0 && carriers[0].ID != 0 {
		status.AirlineId = carriers[0].ID
	}

	station := objects.Station.ByCode(strings.ToUpper(status.Airport))
	if station == nil {
		return xerrors.Errorf("invalid airport: %w", unknownAirportCodeError{status.Airport})
	}
	status.Airport = station.Code()

	if status.ReceivedAt == 0 {
		return xerrors.Errorf("got a status without received at")
	}
	status.FlightNumber = strings.TrimLeft(strings.TrimSpace(status.FlightNumber), "0")
	status.TimeScheduled = scheduledTimeNorm
	status.TimeActual = actualTimeNorm
	status.Airport = strings.ToUpper(status.Airport)
	return nil

}

func (status *tStatus) GetUpdateLogRecord(result updatestatuslog.RESULT) updatestatuslog.Record {
	return updatestatuslog.Record{
		Unixtime:            time.Now().Unix(),
		ReceivedAt:          time.Unix(status.ReceivedAt, 0).UTC().Format("2006-01-02 15:04:05"),
		AirportCode:         status.Airport,
		AirlineID:           status.AirlineId,
		AirlineCode:         status.AirlineCode,
		FlightNumber:        status.FlightNumber,
		FlightDate:          status.FlightDate,
		Direction:           status.Direction,
		TimeActual:          status.TimeActual,
		TimeScheduled:       status.TimeScheduled,
		Status:              status.Status,
		Gate:                status.Gate,
		Terminal:            status.Terminal,
		CheckInDesks:        status.CheckInDesks,
		BaggageCarousels:    status.BaggageCarousels,
		Source:              status.Source,
		Result:              string(result),
		Diverted:            status.Diverted,
		DivertedAirportCode: status.DivertedAirportCode,

		MessageID: status.MessageId,
		StatusID:  status.StatusId,
	}
}
