package status

import (
	"reflect"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects/model"
	"a.yandex-team.ru/travel/library/go/errutil"
)

type statusStatementValues struct {
	AirlineID      int64     `json:"airlineId"`
	AirlineCode    string    `json:"airlineCode"`
	FlightNumber   string    `json:"flightNumber"`
	LegNumber      int16     `json:"legNumber"`
	FlightDate     string    `json:"flightDate"`
	StatusSourceID int16     `json:"statusSourceId"`
	CreatedAtUTC   time.Time `json:"createdAtUTC"`
	UpdatedAtUTC   time.Time `json:"updatedAtUTC"`

	DTimeActual          *string   `json:"{{direction}}TimeActual"`
	DTimeScheduled       *string   `json:"{{direction}}TimeScheduled"`
	DStatus              string    `json:"{{direction}}Status"`
	DGate                string    `json:"{{direction}}Gate"`
	DTerminal            string    `json:"{{direction}}Terminal"`
	DAirport             string    `json:"{{direction}}Airport"`
	DDiverted            bool      `json:"{{direction}}Diverted"`
	DDivertedAirportCode string    `json:"{{direction}}DivertedAirportCode"`
	DCreatedAtUTC        time.Time `json:"{{direction}}CreatedAtUTC"`
	DReceivedAtUTC       time.Time `json:"{{direction}}ReceivedAtUTC"`
	DUpdatedAtUTC        time.Time `json:"{{direction}}UpdatedAtUTC"`
	DRoutePointFrom      string    `json:"{{direction}}RoutePointFrom"`
	DRoutePointTo        string    `json:"{{direction}}RoutePointTo"`
	RoutePointFrom       string    `json:"RoutePointFrom"`
	RoutePointTo         string    `json:"RoutePointTo"`

	CheckInDesks     string `json:"CheckInDesks"`
	BaggageCarousels string `json:"BaggageCarousels"`
}

var statusStatementValuesFields []string

func (v statusStatementValues) fields() []string {
	if len(statusStatementValuesFields) != 0 {
		return statusStatementValuesFields
	}
	var tags []string

	t := reflect.TypeOf(&v).Elem()
	for i := 0; i < t.NumField(); i++ {
		tag := t.Field(i).Tag.Get("json")
		if len(tag) == 0 {
			continue
		}
		splittag := strings.Split(tag, ",")
		tag = splittag[0]
		if len(tag) == 0 {
			continue
		}
		tags = append(tags, tag)

	}
	statusStatementValuesFields = tags
	return tags
}

func (v statusStatementValues) values() []interface{} {
	var tags []interface{}

	t := reflect.TypeOf(&v).Elem()
	for i := 0; i < t.NumField(); i++ {

		tag := t.Field(i).Tag.Get("json")
		if len(tag) == 0 {
			continue
		}
		splittag := strings.Split(tag, ",")
		tag = splittag[0]
		if len(tag) == 0 {
			continue
		}
		tags = append(tags, reflect.ValueOf(v).Field(i).Interface())

	}
	return tags
}

func (v statusStatementValues) pk(dir string) statusStatementDPrimaryKey {
	return statusStatementDPrimaryKey{
		statusStatementPrimaryKey: statusStatementPrimaryKey{
			AirlineCode:    v.AirlineCode,
			FlightNumber:   v.FlightNumber,
			FlightDate:     v.FlightDate,
			LegNumber:      v.LegNumber,
			StatusSourceID: v.StatusSourceID,
		},
		Direction: dir,
	}
}

type statusStatementPrimaryKey struct {
	AirlineCode    string
	FlightNumber   string
	FlightDate     string
	LegNumber      int16
	StatusSourceID int16
}
type statusStatementDPrimaryKey struct {
	statusStatementPrimaryKey
	Direction string
}

type sqlStatement struct {
	name string
	sql  string
}

var directionStatementCache = make(map[direction.Direction]sqlStatement)

func directionStatement(dir direction.Direction) (name string, sql string) {
	if stmt, exists := directionStatementCache[dir]; exists {
		return stmt.name, stmt.sql
	} else {
		defer func(dir direction.Direction, name, sql *string) {
			directionStatementCache[dir] = sqlStatement{*name, *sql}
		}(dir, &name, &sql)
	}
	statementNameTemplate := "{{direction}}_status"
	statusFields := strings.Join(statusStatementValues{}.fields(), ", ")

	statusFieldSequence := func(s []string) string {
		sequenceNumberStrings := make([]string, 0, len(s))
		for i := 0; i < len(s); i++ {
			sequenceNumberStrings = append(sequenceNumberStrings, "$"+strconv.Itoa(i+1))
		}
		return strings.Join(sequenceNumberStrings, ", ")
	}(statusStatementValues{}.fields())
	// TODO(mikhailche): PRO tip: use alembic revision number for seamless on-conflict statement switch
	sqlTemplateInsertOnConflictUpdate := `
INSERT INTO flight_status (` + statusFields + `)
VALUES (` + statusFieldSequence + `) ON CONFLICT (
	airlineCode, flightNumber, flightDate, legNumber, statusSourceID
) DO UPDATE	SET

updatedAtUTC = EXCLUDED.UpdatedAtUTC,

{{direction}}TimeActual=EXCLUDED.{{direction}}TimeActual,
{{direction}}TimeScheduled=EXCLUDED.{{direction}}TimeScheduled,
{{direction}}Status=EXCLUDED.{{direction}}Status,
{{direction}}Gate=EXCLUDED.{{direction}}Gate,
{{direction}}Terminal=EXCLUDED.{{direction}}Terminal,
{{direction}}Airport=EXCLUDED.{{direction}}Airport,
{{direction}}Diverted=EXCLUDED.{{direction}}Diverted,
{{direction}}DivertedAirportCode=EXCLUDED.{{direction}}DivertedAirportCode,
{{direction}}CreatedAtUTC=(case when flight_status.{{direction}}CreatedAtUTC is null then EXCLUDED.{{direction}}CreatedAtUTC else flight_status.{{direction}}CreatedAtUTC end),
{{direction}}ReceivedAtUTC=EXCLUDED.{{direction}}ReceivedAtUTC,
{{direction}}UpdatedAtUTC=EXCLUDED.{{direction}}UpdatedAtUTC,
{{direction}}RoutePointFrom=EXCLUDED.{{direction}}RoutePointFrom,
{{direction}}RoutePointTo=EXCLUDED.{{direction}}RoutePointTo`
	sqlTemplateUpdateCheckInDesks := `CheckInDesks=(case when EXCLUDED.CheckInDesks is null then flight_status.CheckInDesks else EXCLUDED.CheckInDesks end)`
	sqlTemplateUpdateBaggageCarousels := `BaggageCarousels=(case when EXCLUDED.BaggageCarousels is null then flight_status.BaggageCarousels else EXCLUDED.BaggageCarousels end)`
	sqlTemplateWhere := `WHERE flight_status.{{direction}}ReceivedAtUTC IS NULL OR flight_status.{{direction}}ReceivedAtUTC <= EXCLUDED.{{direction}}ReceivedAtUTC`
	sqlTemplateReturning := `RETURNING airlineCode, flightNumber, flightDate, '{{direction}}', {{direction}}Airport`
	if dir == direction.ARRIVAL {
		sqlTemplateInsertOnConflictUpdate = strings.Join(
			[]string{sqlTemplateInsertOnConflictUpdate, sqlTemplateUpdateBaggageCarousels},
			",\n",
		)
	} else if dir == direction.DEPARTURE {
		sqlTemplateInsertOnConflictUpdate = strings.Join(
			[]string{sqlTemplateInsertOnConflictUpdate, sqlTemplateUpdateCheckInDesks},
			",\n",
		)
	}

	sqlTemplate := strings.Join([]string{sqlTemplateInsertOnConflictUpdate, sqlTemplateWhere, sqlTemplateReturning}, "\n")

	replacer := strings.NewReplacer("{{direction}}", dir.String())
	name = replacer.Replace(statementNameTemplate)
	sql = replacer.Replace(sqlTemplate)
	return
}

const unknownLeg = 0
const incorrectLeg = -1

type tStatusWithArguments struct {
	status    *tStatus
	values    statusStatementValues
	arguments []interface{}
}

type statusWithError struct {
	status *tStatus
	err    error
}

type void struct{}

var empty = void{}

func prepareStatusArguments(statuses Statuses, objects *objects.Objects) ([]tStatusWithArguments, []statusWithError) {
	statusesArguments := make([]tStatusWithArguments, 0, len(statuses))
	duplicateTracker := make(map[statusStatementDPrimaryKey]void)
	var argErrs []statusWithError
	for _, status := range statuses {
		statusValues, err := preparedStatementArguments(status, objects)
		if err != nil {
			argErrs = append(argErrs, statusWithError{status, err})
			continue
		}
		primaryKey := statusValues.pk(status.Direction)
		if _, keyExists := duplicateTracker[primaryKey]; !keyExists {
			duplicateTracker[primaryKey] = empty
			statusesArguments = append(
				statusesArguments,
				tStatusWithArguments{status, statusValues, statusValues.values()},
			)
		}
	}
	return statusesArguments, argErrs
}

func preparedStatementArguments(status *tStatus, objects *objects.Objects) (statusStatementValues, error) {
	var (
		statusSourceID  int16
		leg             int16
		flightDeparture dtutil.StringDate
		dir             direction.Direction
		routePointFrom  string
		routePointTo    string
		err             error

		statementValues statusStatementValues
	)
	defer errutil.Wrap(&err, "preparedStatementArguments")
	{
		statusSourceObj := objects.StatusSource.ByName(status.Source)
		if statusSourceObj == nil {
			return statementValues, unknownStatusSourceError{status.Source}
		}
		statusSourceID = statusSourceObj.ID
	}
	{
		dir, err = direction.FromString(status.Direction)
		if err != nil {
			return statementValues, err
		}
		airportAsStation := objects.Station.ByCode(strings.ToUpper(status.Airport))
		if airportAsStation == nil {
			return statementValues, unknownAirportCodeError{status.Airport}
		}
		for _, carrier := range objects.Carrier.ByCode(status.AirlineCode) {
			leg, flightDeparture, err = objects.FlightLeg.FlightLeg(
				carrier.ID, status.FlightNumber, dtutil.StringDate(status.FlightDate), dir,
				airportAsStation.ID,
			)
			if leg > 0 {
				break
			}
		}
		if err != nil {
			return statementValues, err
		}
		if leg == unknownLeg {
			leg = incorrectLeg
		}
		if flightDeparture == "" {
			flightDeparture = dtutil.StringDate(status.FlightDate)
		}

		routePointFrom = GetRoutePoint(status.RoutePointFrom, objects)
		routePointTo = GetRoutePoint(status.RoutePointTo, objects)
	}
	now := time.Now().UTC()

	return statusStatementValues{
		AirlineID:            status.AirlineId,
		AirlineCode:          status.AirlineCode,
		FlightNumber:         status.FlightNumber,
		LegNumber:            leg,
		FlightDate:           string(flightDeparture),
		StatusSourceID:       statusSourceID,
		CreatedAtUTC:         now,
		UpdatedAtUTC:         now,
		DTimeActual:          strutil.EmptyToNil(status.TimeActual),
		DTimeScheduled:       strutil.EmptyToNil(status.TimeScheduled),
		DStatus:              status.Status,
		DGate:                status.Gate,
		DTerminal:            status.Terminal,
		DAirport:             status.Airport,
		DDiverted:            status.Diverted,
		DDivertedAirportCode: status.DivertedAirportCode,
		DCreatedAtUTC:        now,
		DReceivedAtUTC:       time.Unix(status.ReceivedAt, 0).UTC(),
		DUpdatedAtUTC:        now,
		DRoutePointFrom:      routePointFrom,
		RoutePointFrom:       routePointFrom,
		DRoutePointTo:        routePointTo,
		RoutePointTo:         routePointTo,
		CheckInDesks:         status.CheckInDesks,
		BaggageCarousels:     status.BaggageCarousels,
	}, nil
}

func GetRoutePoint(statusRoutePoint string, objects *objects.Objects) string {
	statusInUpper := strings.ToUpper(statusRoutePoint)
	pointAsStation := objects.Station.ByCode(statusInUpper)
	if pointAsStation != nil {
		return statusInUpper
	}
	stopPoint := objects.StopPoint.ByCode(statusRoutePoint)
	if stopPoint != nil {
		if stopPoint.StationID >= 0 {
			station := objects.Station.ByID(model.StationID(stopPoint.StationID))
			if station != nil {
				return station.Code()
			}
		}
	}
	return statusRoutePoint
}
