package storage

import (
	"path"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/library/go/core/log"
	arczap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/db"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects/combined"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/settings"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/status"
	"a.yandex-team.ru/travel/library/go/logging"
)

// New инициализует и возвращает экземпляр структуры StorageAPI
func New(c settings.Config) *storage {
	var err error
	var cluster db.Cluster
	err = backoff.RetryNotify(
		func() error {
			var clusterErr error
			cluster, clusterErr = db.NewCluster(&c.PG)
			return clusterErr
		}, backoff.WithMaxRetries(backoff.NewConstantBackOff(time.Second), 3),
		func(e error, d time.Duration) {
			logger.Logger().Info(
				"Connection to database failed. Retrying",
				log.Any("duration", d),
				log.Error(e),
			)
		},
	)

	if err != nil {
		logger.Logger().Fatal("Cannot connect to database", log.Error(err))
	}
	objs := combined.New(objects.Config{
		Cluster:        cluster,
		UpdateInterval: c.Parameters.CacheUpdateInterval,
		PrecacheWindow: c.Parameters.PrecacheWindow,
	})

	var updateLogger *arczap.Logger
	if !c.YtLogging.UpdateStatusLogDisabled {
		updateLogger, err = logging.NewYtFileLogger(log.InfoLevel, path.Join(c.YtLogging.YtLogPath, "yt", c.YtLogging.UpdateStatusLogName))
		if err != nil {
			logger.Logger().Fatal("can not create YT logger")
		}
	} else {
		logger.Logger().Warn("Update status log is disabled by environment")
		updateLogger, err = arczap.NewDeployLogger(log.FatalLevel)
		if err != nil {
			logger.Logger().Fatal("Cannot create placeholder logger for `update status` log", log.Error(err))
		}
	}

	statusUpdater := status.NewStatusUpdater(
		status.UpdaterConfig{
			Parallelism:   c.Parameters.UpdateStatusWorkers,
			BatchSenderFn: func() status.BatchSender { return cluster.RW() },
			Objects:       objs,
			UpdateLogger:  updateLogger,
		},
	)

	application := storage{
		Conf:          &c,
		PGCluster:     cluster,
		statusUpdater: statusUpdater,
	}
	return &application
}
