# coding=utf-8
from __future__ import unicode_literals

from datetime import datetime
import tempfile

import paramiko

from travel.avia.shared_flights.tasks.amadeus_parser.flights_parser import AmadeusFlightsParser
from travel.library.python.dicts import file_util


SFTP_FOLDER = './1A_TI-YANDEX_DATA/SCHEDULE/'
TARGET_FILE_PREFIX = 'StandardSA_F'


class AmadeusFetchResult(object):
    """ Result of Amadeus files fetching. """

    def __init__(self, data_file_timestamp, flight_bases_file, flight_patterns_file, codeshares_file):
        self._data_file_timestamp = data_file_timestamp
        self._flight_bases_file = flight_bases_file
        self._flight_patterns_file = flight_patterns_file
        self._codeshares_file = codeshares_file

    @property
    def data_file_timestamp(self):
        return self._data_file_timestamp

    @property
    def flight_bases_file(self):
        return self._flight_bases_file

    @property
    def flight_patterns_file(self):
        return self._flight_patterns_file

    @property
    def codeshares_file(self):
        return self._codeshares_file


class AmadeusFetcher(object):
    """ Fetches CSV files from Amadeus over sftp. """

    def __init__(self, hostname, username, password, logger):
        self._hostname = hostname
        self._username = username
        self._password = password
        self._logger = logger

    def fetch(self, last_parsed_date):
        zip_csv_file = None
        with paramiko.SSHClient() as ssh_client:
            ssh_client.set_missing_host_key_policy(paramiko.AutoAddPolicy())

            self._logger.info('About to connect')
            ssh_client.connect(
                hostname=self._hostname,
                port=9022,
                username=self._username,
                password=self._password,
                look_for_keys=False,
                allow_agent=False,
                timeout=2700,
            )

            data_file_timestamp = None
            self._logger.info('About to open sftp')
            with ssh_client.open_sftp() as sftp:
                csv_file_name = None
                for sftp_attr in sftp.listdir_attr(SFTP_FOLDER):
                    if not sftp_attr.filename.startswith(TARGET_FILE_PREFIX):
                        continue
                    if sftp_attr.st_mtime:
                        data_file_timestamp = datetime.utcfromtimestamp(sftp_attr.st_mtime).strftime('%Y-%m-%d %H:%M:%S')
                        csv_file_name = '{}{}'.format(SFTP_FOLDER, sftp_attr.filename)
                    self._logger.info('listdir: {}'.format(sftp_attr))
                if not data_file_timestamp:
                    raise Exception(
                        'Unable to locate files starting with {} on {}. Please update the task parameters.'.format(
                            TARGET_FILE_PREFIX,
                            self._ftp_host,
                        )
                    )
                self._logger.info('The latest target file found is from {}'.format(data_file_timestamp))
                if last_parsed_date is not None and last_parsed_date >= data_file_timestamp:
                    self._logger.info('Last parsed date {} is already no older than the current one {}'.format(
                        last_parsed_date, data_file_timestamp))
                    return None

                # download schedule
                self._logger.info('Creating temp file')
                zip_csv_file = tempfile.NamedTemporaryFile().name
                self._logger.info('Starting download')
                sftp.get(csv_file_name, zip_csv_file)

        # parse flights
        flights_parser = AmadeusFlightsParser(self._logger)
        flights_parser.parse_flights_zip(zip_csv_file)

        # store results
        flight_bases_file = tempfile.NamedTemporaryFile().name
        flight_patterns_file = tempfile.NamedTemporaryFile().name
        codeshares_file = tempfile.NamedTemporaryFile().name
        self._logger.info('Temporary stored flights_bases at %s', flight_bases_file)
        self._logger.info('Temporary stored flights_patterns at %s', flight_patterns_file)
        self._logger.info('Temporary stored codeshares at %s', codeshares_file)

        with open(flight_bases_file, 'wb') as fb_file, open(flight_patterns_file, 'wb') as fp_file, open(codeshares_file, 'wb') as cs_file:
            flights_parser.list_flights(
                on_flight_base=lambda fb: file_util.write_binary_string(fb_file, fb.SerializeToString()),
                on_flight_pattern=lambda fp: file_util.write_binary_string(fp_file, fp.SerializeToString()),
                on_codeshare=lambda cs: file_util.write_binary_string(cs_file, cs.SerializeToString()),
            )

        self._logger.info('Done writing flights')

        return AmadeusFetchResult(
            data_file_timestamp,
            flight_bases_file,
            flight_patterns_file,
            codeshares_file,
        )
