# coding=utf-8
from __future__ import unicode_literals

from collections import Counter, namedtuple


FlyoutKey = namedtuple('FlyoutKey', ['carrier', 'flight_number', 'dep_date', 'dep_station', 'arrival_station'])
Flyout = namedtuple('Flyout', ['carrier', 'flight_number', 'dep_date'])


class DoubleFlyoutsFilter(object):
    """ Counts the number of times each segment has been listed for one day """

    def __init__(self):
        self._segments_per_day_counts = Counter()
        self._duplicate_flyouts = set()

    def cache_segment_on_date(self, flight_segment):
        if not flight_segment.route or not flight_segment.operating_carrier or not flight_segment.operating_flight_code:
            return
        # After split('-'), last point is going to have '\n' suffix. It's okay to not trim it,
        # since the segment can not start with the last point of the route
        stops = flight_segment.route.split('-')
        if len(stops) >= 3 or flight_segment.stops != '0':  # only cache segments, not multi-segments
            return

        self._segments_per_day_counts[self._get_key(flight_segment)] += 1

    def _get_key(self, flight_segment):
        return FlyoutKey(
            flight_segment.operating_carrier,
            flight_segment.operating_flight_code,
            flight_segment.dep_date,
            flight_segment.board_airport,
            flight_segment.off_airport,
        )

    def extract_duplicate_flyouts(self):
        self._duplicate_flyouts = set()
        for k, v in self._segments_per_day_counts.items():
            if v < 2:
                continue
            self._duplicate_flyouts.add(Flyout(k.carrier, k.flight_number, k.dep_date))

    def is_duplicate(self, carrier, flight_number, dep_date):
        return Flyout(carrier, flight_number, dep_date) in self._duplicate_flyouts

    def flights_count(self):
        cnt = set()
        for flyout in self._duplicate_flyouts:
            cnt.add('{}.{}'.format(flyout.carrier, flyout.flight_number))
        return len(cnt)

    def flights_and_days_count(self):
        return len(self._duplicate_flyouts)
