# coding=utf-8
from __future__ import unicode_literals

from collections import namedtuple
import six


class FlightSegment(namedtuple(
    'FlightSegment',
    [
        'operating_carrier',
        'operating_flight_code',
        'stops',
        'dep_date',
        'dep_time',
        'dep_weekday',
        'dep_week',
        'arrival_date',
        'arrival_time',
        'arrival_weekday',
        'arrival_week',
        'board_airport',
        'board_city',
        'board_country',
        'board_region',
        'board_terminal',
        'off_airport',
        'off_city',
        'off_country',
        'off_region',
        'off_terminal',
        'direct_distance',
        'total_capacity',
        'first_total_capacity',
        'business_total_capacity',
        'economy_total_capacity',
        'premiumeco_total_capacity',
        'codeshare_flights',
        'aircraft_type',
        'aircraft_family',
        'route',
        'leg_sequence_number',
    ],
)):
    def get_leg_number(self):
        # type: () -> int
        try:
            return int(self.leg_sequence_number)
        except:
            return 0

    def get_dep_time_int(self):
        try:
            return int(self.dep_time)
        except:
            return -1


class FlightSegmentFactory:
    """ Converts lines from the incoming Amadeus csv file into flight segments """

    def __init__(self, headers):
        # type: (list) -> None
        self._headers = headers

    def new_flight_segment(self, values):
        data = {}
        for header, value in zip(self._headers, values):
            data[header] = value
        return FlightSegment(
            operating_carrier=data.get('operating_carrier'),
            operating_flight_code=data.get('operating_flight_code'),
            stops=data.get('stops'),
            dep_date=data.get('dep_date'),
            dep_time=data.get('dep_time'),
            dep_weekday=data.get('dep_weekday'),
            dep_week=data.get('dep_week'),
            arrival_date=data.get('arrival_date'),
            arrival_time=data.get('arrival_time'),
            arrival_weekday=data.get('arrival_weekday'),
            arrival_week=data.get('arrival_week'),
            board_airport=data.get('board_airport'),
            board_city=data.get('board_city'),
            board_country=data.get('board_country'),
            board_region=data.get('board_region'),
            board_terminal=data.get('board_terminal'),
            off_airport=data.get('off_airport'),
            off_city=data.get('off_city'),
            off_country=data.get('off_country'),
            off_region=data.get('off_region'),
            off_terminal=data.get('off_terminal'),
            direct_distance=data.get('direct_distance'),
            total_capacity=data.get('total_capacity'),
            first_total_capacity=data.get('first_total_capacity'),
            business_total_capacity=data.get('business_total_capacity'),
            economy_total_capacity=data.get('economy_total_capacity'),
            premiumeco_total_capacity=data.get('premiumeco_total_capacity'),
            codeshare_flights=data.get('codeshare_flights'),
            aircraft_type=data.get('aircraft_type'),
            aircraft_family=data.get('aircraft_family'),
            route=data.get('route'),
            leg_sequence_number=data.get('leg_sequence_number'),
        )

    def parse_flight_segment(self, line):
        # type: (str) -> FlightSegment
        if not line:
            return None
        parts = six.ensure_str(line).split('^')
        if len(parts) != len(self._headers):
            raise Exception('Line {} does not have the same number of parts as file headers, no way to parse the file'.format(line))
        return self.new_flight_segment(parts)
