import logging
from typing import Callable, Generator

from sqlalchemy.orm import Session

from travel.avia.shared_flights.tasks.monitoring.db.import_delay_monitor import LastImportedInfoMonitor
from travel.avia.shared_flights.tasks.monitoring.db.lock_monitor import LockMonitor
from travel.avia.shared_flights.tasks.monitoring.db.tables_size import TableSizeMonitor
from travel.avia.shared_flights.tasks.monitoring.db.metric import Metric

logger = logging.getLogger(__name__)


def collect_all(session_factory: Callable[[], Session]) -> Generator[Metric, None, None]:
    """This is a wrapper that returns metrics from both metric collectors with right labels"""
    logger.info('Collecting metrics using provided database session factory')
    single_stat: Metric
    for single_stat in LastImportedInfoMonitor(session_factory).get_metrics():
        single_stat.labels.update({'component': 'data-importer'})
        yield single_stat

    yield from LockMonitor(session_factory).get_metrics()

    yield from TableSizeMonitor(session_factory).get_metrics()

    logger.info('Done collecting all metrics')


def collect_and_send(session_factory, solomon_push_reporter):
    logger.info('Collecting and sending metrics')
    single_stat: Metric
    for single_stat in collect_all(session_factory):
        solomon_push_reporter.set_value(
            single_stat.sensor,
            single_stat.value,
            single_stat.labels,
        )
        logger.info('Set solomon metrics value: %r', single_stat)
    logger.info('Done collecting and sending metrics')
