from datetime import datetime

from sqlalchemy.orm import Session
from typing import Callable, Generator

from travel.avia.shared_flights.tasks.monitoring.db.metric import Metric
from travel.avia.shared_flights.lib.python.db_models.last_imported_info import LastImportedInfo


class LastImportedInfoMonitor:
    def __init__(self, session_factory: Callable[[], Session]):
        self.session_factory = session_factory

    def get_metrics(self) -> Generator[Metric, None, None]:
        for metric in self.get_last_imported_resource_info():
            metric.sensor = 'last_imported_resource.' + metric.sensor
            yield metric

    def get_last_imported_resource_info(self) -> Generator[Metric, None, None]:
        session = self.session_factory()
        try:
            imported_info: LastImportedInfo
            now = datetime.now()
            for imported_info in session.query(LastImportedInfo):
                resource_type = imported_info.resource_type
                imported_date = imported_info.imported_date
                updated_at = imported_info.updated_at

                yield Metric(
                    sensor='since_imported',
                    labels={
                        'resource_type': resource_type,
                    },
                    value=(now - imported_date).total_seconds(),
                )
                yield Metric(
                    sensor='since_updated',
                    labels={
                        'resource_type': resource_type,
                    },
                    value=(now - updated_at).total_seconds(),
                )
        finally:
            session.close()
