import logging

from datetime import datetime

from sqlalchemy.orm import Session
from typing import Callable, Generator

from travel.avia.shared_flights.tasks.monitoring.db.metric import Metric
from travel.avia.shared_flights.lib.python.db_models.db_lock import DbLock

logger = logging.getLogger(__name__)


class LockMonitor:
    def __init__(self, session_factory: Callable[[], Session]):
        self.session_factory = session_factory

    def get_metrics(self) -> Generator[Metric, None, None]:
        for metric in self.get_lock_status():
            metric.sensor = 'db_lock.' + metric.sensor
            yield metric

    def get_lock_status(self) -> Generator[Metric, None, None]:
        session = self.session_factory()
        try:
            now = datetime.now()

            db_lock: DbLock
            for db_lock in session.query(DbLock):
                logger.debug("Db lock values: %r", db_lock)
                yield Metric(
                    sensor='since_expired',
                    labels={
                        'lock_type': db_lock.lock_type,
                    },
                    value=(now - db_lock.expires_at).total_seconds(),
                )
                yield Metric(
                    sensor='since_updated',
                    labels={
                        'lock_type': db_lock.lock_type,
                    },
                    value=(now - db_lock.updated_at).total_seconds(),
                )
        finally:
            session.close()
