import logging

from solomon import OAuthProvider, BasePushApiReporter
from sqlalchemy import create_engine
from sqlalchemy.orm import sessionmaker
from travel.avia.shared_flights.lib.python.db.engine import conn_string

logger = logging.getLogger(__name__)


def solomon_push_reporter_maker(environment, token):
    logger.info(
        'Creating solomon push reporter for environment %s. Token is provided: %s',
        environment,
        True if token else False,
    )
    SOLOMON_PUSH_API_REPORTER_CONFIG_FOR_ENVIRONMENT = {
        'development': {
            'project': 'avia',
            'cluster': 'shared-flights_testing',
            'service': 'app',
            'common_labels': {'environment': 'development'},
            'url': 'http://solomon.yandex.net',
        },
        'unstable': {
            'project': 'avia',
            'cluster': 'shared-flights_testing',
            'service': 'app',
            'common_labels': {'environment': 'unstable'},
            'url': 'http://solomon.yandex.net',
        },
        'testing': {
            'project': 'avia',
            'cluster': 'shared-flights_testing',
            'service': 'app',
            'common_labels': {'environment': 'testing'},
            'url': 'http://solomon.yandex.net',
        },
        'production': {
            'project': 'avia',
            'cluster': 'shared-flights_production',
            'service': 'app',
            'common_labels': {'environment': 'production'},
            'url': 'http://solomon.yandex.net',
        },
    }
    if environment not in SOLOMON_PUSH_API_REPORTER_CONFIG_FOR_ENVIRONMENT:
        raise UnsupportedEnvironmentError(f'Environment {environment} is not supported')

    auth_provider = None
    if token:
        auth_provider = OAuthProvider(token)

    config = SOLOMON_PUSH_API_REPORTER_CONFIG_FOR_ENVIRONMENT[environment]
    logger.info('Solomon push reporter with config %s is about to be created', config)
    return BasePushApiReporter(
        project=config['project'],
        cluster=config['cluster'],
        service=config['service'],
        url=config['url'],
        common_labels=config['common_labels'],
        auth_provider=auth_provider,
    )


def pg_cluster_session(environment, password, echo=True):
    POSTGRESQL_CONFIG_FOR_ENVIRONMENT = {
        'development': {
            'cluster': 'mdbopcn7a3e8h7sh15db',
            'port': '6432',
            'user': 'avia',
            'database': 'flights',
        },
        'unstable': {
            'cluster': 'mdb9sssbmtcje8gtvlrc',
            'port': '6432',
            'user': 'avia',
            'database': 'shared-flights-unstable',
        },
        'testing': {
            'cluster': 'mdb9sssbmtcje8gtvlrc',
            'port': '6432',
            'user': 'avia',
            'database': 'shared-flights',
        },
        'production': {
            'cluster': 'mdbgrlcnlbkadhqbv6e6',
            'port': '6432',
            'user': 'avia',
            'database': 'shared-flights',
        },
    }
    if environment not in POSTGRESQL_CONFIG_FOR_ENVIRONMENT:
        raise UnsupportedEnvironmentError(f'Environment {environment} is not supported')

    config = POSTGRESQL_CONFIG_FOR_ENVIRONMENT[environment]
    return sessionmaker(
        bind=create_engine(
            conn_string(config['user'], password, config['database'], config['port'], config['cluster']),
            echo=echo,
        ),
    )


class UnsupportedEnvironmentError(ValueError):
    pass
