package internal

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"time"

	"github.com/valyala/fastjson"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
)

// Borrowed from travel/hotels/tools/boy_hotels_checker/pkg/yql/yql.go

type YQLStatus string

const yqlRoot = "https://yql.yandex.net/api/v2/operations"

const (
	YQLStatusIdle      YQLStatus = "IDLE"
	YQLStatusPending   YQLStatus = "PENDING"
	YQLStatusRunning   YQLStatus = "RUNNING"
	YQLStatusCompleted YQLStatus = "COMPLETED"
	YQLStatusAborting  YQLStatus = "ABORTING"
	YQLStatusAborted   YQLStatus = "ABORTED"
	YQLStatusError     YQLStatus = "ERROR"
	YQLStatusUnknown   YQLStatus = "UNKNOWN"
)

type YqlOperation struct {
	query   string
	id      string
	token   string
	params  map[string]string
	Status  YQLStatus
	Results []map[string]Value
	logger  log.Logger
}

type yqlRequest struct {
	Content    string            `json:"content"`
	Action     string            `json:"action"`
	ActionType string            `json:"type"`
	Parameters map[string]string `json:"parameters"`
}

type Any interface{}

type Value struct {
	scalar Any
	array  []Any
}

func NewYqlOperation(query string, params map[string]interface{}, token string, logger log.Logger) (*YqlOperation, error) {
	yqlParams, err := convertParamsToYQLParams(params)
	if err != nil {
		return nil, err
	}
	return &YqlOperation{
		query:  query,
		params: yqlParams,
		token:  token,
		Status: YQLStatusUnknown,
		logger: logger,
	}, nil
}

func convertParamsToYQLParams(params map[string]interface{}) (map[string]string, error) {
	result := make(map[string]string)
	for key, value := range params {
		yqlValue := make(map[string]string)
		yqlValue["Data"] = fmt.Sprintf("%v", value)
		serialized, err := json.Marshal(yqlValue)
		if err != nil {
			return nil, err
		}
		result[key] = string(serialized)
	}
	return result, nil
}

func (o *YqlOperation) Start() error {
	if o.query == "" {
		o.logger.Fatal("No query specified")
	}
	if o.id != "" {
		o.logger.Fatal("Query already started")
	}
	header := http.Header{}
	header.Set("Authorization", fmt.Sprintf("OAuth %s", o.token))
	resp, err := httpPostJSON(yqlRoot, yqlRequest{
		Content:    o.query,
		Parameters: o.params,
		Action:     "RUN",
		ActionType: "SQLv1",
	}, header)
	if err != nil {
		return xerrors.Errorf("unable to start yql query: %w %v", err, resp)
	}
	o.id = string(resp.GetStringBytes("id"))
	o.Status = YQLStatus(resp.GetStringBytes("status"))
	return nil
}

func (o *YqlOperation) Wait() error {
	for o.Status != YQLStatusCompleted && o.Status != YQLStatusAborted && o.Status != YQLStatusError {
		header := http.Header{}
		header.Set("Authorization", fmt.Sprintf("OAuth %s", o.token))
		resp, err := httpGet(yqlRoot+"/"+o.id, url.Values{}, header)
		if err != nil {
			return xerrors.Errorf("unable to get yql operation: %w", err)
		}
		o.Status = YQLStatus(resp.GetStringBytes("status"))
		if o.Status != YQLStatusCompleted && o.Status != YQLStatusAborted && o.Status != YQLStatusError {
			time.Sleep(10 * time.Second)
		}
	}
	if o.Status == YQLStatusCompleted {
		header := http.Header{}
		header.Set("Authorization", fmt.Sprintf("OAuth %s", o.token))
		args := url.Values{}
		args.Set("filters", "DATA")
		o.logger.Info("YQL results", log.String("operation_id", o.id))
		resp, err := httpGet(yqlRoot+"/"+o.id+"/results", args, header)
		if err != nil {
			return xerrors.Errorf("unale to get yql operation: %w", err)
		}
		results := resp.GetArray("data")
		if len(results) > 1 {
			return xerrors.New("unsupported amount of result sets")
		}
		return nil
	} else {
		return xerrors.Errorf("unexpected yql operation %s status %s", o.id, string(o.Status))
	}
}

func httpPostJSON(url string, payload interface{}, header http.Header) (*fastjson.Value, error) {
	serialized, err := json.Marshal(payload)
	if err != nil {
		return nil, xerrors.Errorf("unable to serialize request: %w", err)
	}
	request, err := http.NewRequest("POST", url, bytes.NewReader(serialized))
	if err != nil {
		return nil, xerrors.Errorf("unable to execute http request: %w", err)
	}
	header.Set("Content-Type", "application/json")
	request.Header = header
	resp, err := http.DefaultClient.Do(request)
	if err != nil {
		return nil, xerrors.Errorf("unable to execute http request: %w", err)
	}
	if resp.StatusCode != 200 {
		var output interface{}
		result, err := ioutil.ReadAll(resp.Body)
		if err == nil {
			output, _ = fastjson.ParseBytes(result)
		} else {
			output = result
		}
		return nil, xerrors.Errorf("http request error: %d %v", resp.StatusCode, output)
	}
	result, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, xerrors.Errorf("unable to read response body: %w", err)
	}
	return fastjson.ParseBytes(result)
}

func httpGet(url string, args url.Values, header http.Header) (*fastjson.Value, error) {
	request, err := http.NewRequest("GET", url, nil)
	if err != nil {
		return nil, xerrors.Errorf("error while building http request: %w", err)
	}
	request.URL.RawQuery = args.Encode()
	request.Header = header
	resp, err := http.DefaultClient.Do(request)
	if err != nil {
		return nil, xerrors.Errorf("unable to execute http request: %w", err)
	}
	if resp.StatusCode != 200 {
		return nil, xerrors.Errorf("http request error: %d", resp.StatusCode)
	}
	result, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, xerrors.Errorf("unable to read response body: %w", err)
	}
	return fastjson.ParseBytes(result)
}
