package main

import (
	"context"
	"os"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/shared_flights/tasks/regular_flights/internal"
	"a.yandex-team.ru/travel/library/go/configuration"
)

type Config struct {
	YQLToken                  string `config:"AVIA_YQL_TOKEN,required"`
	StartDate                 string `config:"AVIA_RF_START_DATE,required"`
	EndDate                   string `config:"AVIA_RF_END_DATE,required"`
	YTTablePath               string `config:"AVIA_RF_OUTPUT_YT_TABLE"`
	YTIntermediateTableFolder string `config:"AVIA_RF_TEMP_YT_FOLDER"`
	YTProxy                   string `config:"AVIA_RF_YT_PROXY"`
	Limit                     int    `config:"AVIA_RF_LIMIT"`
}

func main() {
	err := run()
	if err != nil {
		os.Exit(1)
	}
}

func run() error {
	defer func() {
		_ = logger.Logger().L.Sync()
	}()

	loader := configuration.NewDefaultConfitaLoader()
	config := Config{
		YTProxy:                   "hahn",
		YTIntermediateTableFolder: "//home/avia/tmp",
	}
	if err := loader.Load(context.Background(), &config); err != nil {
		logger.Logger().Error("Cannot load config", log.Error(err))
		return err
	}

	logger.Logger().Info("Start building list of regular flights")

	regularFlightsProcessor := internal.RegularFlightsProcessor{
		Token:                     config.YQLToken,
		StartDate:                 config.StartDate,
		EndDate:                   config.EndDate,
		YTProxy:                   config.YTProxy,
		YTTablePath:               config.YTTablePath,
		YTIntermediateTableFolder: config.YTIntermediateTableFolder,
		Limit:                     config.Limit,
		Logger:                    logger.Logger(),
	}
	err := regularFlightsProcessor.GenerateRegularFlights()
	if err != nil {
		logger.Logger().Error("Error while building regular flights list", log.Error(err))
		return err
	}

	logger.Logger().Info("Done building list of regular flights")
	return nil
}
