# coding=utf-8
from __future__ import unicode_literals

import logging
import six
import ujson
from datetime import datetime

from travel.avia.shared_flights.lib.python.consts.consts import TIME_NOT_SPECIFIED
from travel.library.python.safexml.safe_xml_parser import safe_xml_fromstring
from travel.proto.shared_flights.sirena.routes_pb2 import TRoute, TSirenaFlightPattern, TStopPoint


logger = logging.getLogger(__name__)


class RoutesImporter:
    ''' Parses Sirena airline schedule files into a protobuf. '''

    def parse(self, carrier_to_xml_dict, carriers):
        routes = []
        last_routes_length = 0
        for title_carrier, schedule_file_contents in carrier_to_xml_dict.items():
            result = safe_xml_fromstring(six.ensure_binary(schedule_file_contents))
            for flight_el in result.xpath('//f'):
                route = TRoute()
                route.CarrierCode = six.ensure_str(flight_el.get('c', '').strip())
                route.FlightNumber = six.ensure_str(flight_el.get('n', '').strip())
                carrier = carriers.get(route.CarrierCode)
                if not carrier:
                    logger.error(
                        'Carrier code %s is unknown, title carrier is %s, known carriers are %s',
                        ujson.dumps(route.CarrierCode),
                        ujson.dumps(title_carrier),
                        ujson.dumps(carriers.keys()),
                    )
                    continue

                sirena_flight_patterns = []
                for schedule_el in flight_el.findall('./pe'):
                    sirena_flight_pattern = TSirenaFlightPattern()
                    operating_from_date = schedule_el.get('s', '')
                    sirena_flight_pattern.OperatingFromDate = RoutesImporter.get_date(operating_from_date)
                    operating_until_date = schedule_el.get('e', '')
                    sirena_flight_pattern.OperatingUntilDate = RoutesImporter.get_date(operating_until_date)
                    operating_on_days = schedule_el.get('f', '')
                    sirena_flight_pattern.OperatingOnDays = RoutesImporter.get_days(operating_on_days)
                    sirena_flight_pattern.AircraftModel = six.ensure_str(schedule_el.get('t', ''))

                    if sirena_flight_pattern.OperatingFromDate == 0:
                        logger.error(
                            'OperatingFromDate %s is invalid (schedule %s, flight %s %s)',
                            operating_from_date,
                            ujson.dumps(title_carrier),
                            ujson.dumps(route.CarrierCode),
                            ujson.dumps(route.FlightNumber),
                        )
                        continue

                    if sirena_flight_pattern.OperatingUntilDate == 0:
                        logger.error(
                            'OperatingUntilDate %s is invalid (schedule %s, flight %s %s)',
                            operating_until_date,
                            ujson.dumps(title_carrier),
                            ujson.dumps(route.CarrierCode),
                            ujson.dumps(route.FlightNumber),
                        )
                        continue

                    if sirena_flight_pattern.OperatingOnDays == 0:
                        logger.error(
                            'OperatingOnDays %s is invalid (schedule %s, flight %s %s)',
                            operating_on_days,
                            ujson.dumps(title_carrier),
                            ujson.dumps(route.CarrierCode),
                            ujson.dumps(route.FlightNumber),
                        )
                        continue

                    stop_points = []
                    for stoppoint_el in schedule_el.findall('./p'):
                        stop_point = TStopPoint()
                        stop_point.StationCode = six.ensure_str(stoppoint_el.get('p', ''))
                        stop_point.CityCode = six.ensure_str(stoppoint_el.get('c', ''))
                        stop_point.DepartureTime = RoutesImporter.get_time(stoppoint_el.get('dt'))
                        stop_point.DepartureDayShift = int(stoppoint_el.get('do', 0))
                        stop_point.ArrivalTime = RoutesImporter.get_time(stoppoint_el.get('at'))
                        stop_point.ArrivalDayShift = int(stoppoint_el.get('ao', 0))
                        stop_point.Terminal = six.ensure_str(stoppoint_el.get('term', ''))
                        stop_points.append(stop_point)

                    codeshare_el = schedule_el.find('.//cs')
                    if codeshare_el is not None:
                        operating_carrier_code = six.ensure_str(codeshare_el.get('c', ''))
                        operating_flight_number = six.ensure_str(codeshare_el.get('n', ''))
                        wet_lease = int(codeshare_el.get('f', 0))
                        if operating_carrier_code and (operating_flight_number or wet_lease):
                            sirena_flight_pattern.IsCodeshare = True
                            sirena_flight_pattern.OperatingFlight.CarrierCode = operating_carrier_code
                            if operating_flight_number:
                                sirena_flight_pattern.OperatingFlight.FlightNumber = operating_flight_number
                            elif wet_lease:
                                sirena_flight_pattern.OperatingFlight.FlightNumber = route.FlightNumber

                    sirena_flight_pattern.StopPoints.extend(stop_points)

                    if len(sirena_flight_pattern.StopPoints) == 0:
                        logger.error(
                            'No stop points (schedule %s, flight %s %s)',
                            ujson.dumps(title_carrier),
                            ujson.dumps(route.CarrierCode),
                            ujson.dumps(route.FlightNumber),
                        )
                        continue

                    sirena_flight_patterns.append(sirena_flight_pattern)

                route.FlightPatterns.extend(sirena_flight_patterns)

                if len(route.FlightPatterns) == 0:
                    logger.error(
                        'No flight patterns (schedule %s, flight %s %s)',
                        ujson.dumps(title_carrier),
                        ujson.dumps(route.CarrierCode),
                        ujson.dumps(route.FlightNumber),
                    )
                    continue
                routes.append(route)

            logger.info(
                'Parsed %d routes from Sirena for the schedule of carrier %s',
                len(routes) - last_routes_length,
                ujson.dumps(title_carrier),
            )
            last_routes_length = len(routes)

        return routes

    @staticmethod
    def get_time(time_str):
        if not time_str:
            return TIME_NOT_SPECIFIED

        try:
            hours = int(time_str[:2])
            minutes = int(time_str[2:4])
            return hours * 100 + minutes
        except (TypeError, ValueError):
            return TIME_NOT_SPECIFIED

    @staticmethod
    def get_date(date_str):
        if not date_str:
            return 0

        try:
            date_value = datetime.strptime(date_str, '%d.%m.%Y').date()
            return (date_value.year * 100 + date_value.month) * 100 + date_value.day
        except (TypeError, ValueError):
            return 0

    @staticmethod
    def get_days(days_str):
        if not days_str:
            return 0

        try:
            result = 0
            for c in range(1, 8):
                if str(c) in days_str:
                    result = result * 10 + c
            return result
        except (TypeError, ValueError):
            return 0
