# coding=utf-8
from __future__ import unicode_literals

from collections import namedtuple
from google.protobuf import json_format

from travel.avia.library.python.shared_dicts.rasp import get_repository, ResourceType
from travel.library.python.safexml.safe_xml_parser import safe_xml_fromstring
from travel.proto.dicts.rasp.transport_model_pb2 import TTransportModel
from travel.proto.dicts.rasp.transport_pb2 import TTransport


SirenaTransportModel = namedtuple('SirenaTransportModel', ['code', 'code_en', 'title', 'title_en'])


class TransportModelsImporter:
    """ Parses an Sirena aircraft models data """

    def parse(self, transport_models_file_contents):
        transport_models = []

        for vehicle_el in safe_xml_fromstring(transport_models_file_contents).xpath('//data'):
            code = ''
            code_en = ''
            for code_el in vehicle_el.findall('./code'):
                for name, value in code_el.items():
                    if name.endswith('lang'):
                        if value == 'ru' and code_el.text:
                            code = code_el.text
                        elif value == 'en' and code_el.text:
                            code_en = code_el.text

            title = ''
            title_en = ''
            for name_el in vehicle_el.findall('./name'):
                for name, value in name_el.items():
                    if name.endswith('lang'):
                        if value == 'ru' and name_el.text:
                            title = name_el.text
                        elif value == 'en' and name_el.text:
                            title_en = name_el.text

            transport_models.append(SirenaTransportModel(code, code_en, title, title_en))

        return transport_models

    def compare_with_rasp_dict(self, sirena_transport_models, oauth_token):
        transport_models_repository = get_repository(
            ResourceType.TRAVEL_DICT_RASP_TRANSPORT_MODEL_PROD,
            oauth=oauth_token,
        )
        new_transport_models = []
        transport_models = {}
        for transport_model in transport_models_repository.itervalues():
            transport_type = transport_model.TransportType
            if transport_type != TTransport.EType.TYPE_PLANE and transport_type != TTransport.EType.TYPE_HELICOPTER:
                continue
            transport_models[transport_model.Code] = transport_model
            transport_models[transport_model.CodeEn] = transport_model

        for sirena_transport_model in sirena_transport_models:
            if sirena_transport_model.code and transport_models.get(sirena_transport_model.code):
                continue
            if sirena_transport_model.code_en and transport_models.get(sirena_transport_model.code_en):
                continue
            new_transport_model = TTransportModel()
            new_transport_model.Code = sirena_transport_model.code
            new_transport_model.CodeEn = sirena_transport_model.code_en
            new_transport_model.Title = sirena_transport_model.title
            new_transport_model.TitleEn = sirena_transport_model.title_en
            new_transport_models.append(json_format.MessageToJson(new_transport_model))

        return new_transport_models
