import argparse
import logging
import zipfile

from travel.avia.shared_flights.tasks.ssim_parser.aircraft import AircraftImporter
from travel.avia.shared_flights.tasks.ssim_parser.flights import FlightsImporter
from travel.library.python.dicts import file_util

logging.basicConfig(
    format='%(asctime)s %(levelname)-8s %(message)s',
    level=logging.DEBUG,
    datefmt='%Y-%m-%d %H:%M:%S')
logger = logging.getLogger(__name__)


# Sample run:
# bin-parse/ssim-parser --infile ssim.zip --outfile flights.txt --flight_patterns flight_patterns.txt \
#                       --designated_carriers designated_carriers.txt --flying_carriers flying_carriers.txt \
#                       --codeshares codeshares.txt --datatype flights
def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--infile', required=True)
    parser.add_argument('--outfile', required=True)
    parser.add_argument('--flight_patterns', required=False, help='Required when datatype=flights')
    parser.add_argument('--designated_carriers', required=False, help='Required when datatype=flights')
    parser.add_argument('--flying_carriers', required=False, help='Required when datatype=flights')
    parser.add_argument('--codeshares', required=False, help='Required when datatype=flights')
    parser.add_argument('--datatype', choices=['aircraft', 'flights'], required=True)
    _args = vars(parser.parse_args())

    infile = _args['infile']
    outfile = _args['outfile']
    datatype = _args['datatype']

    if datatype == 'flights':
        flight_patterns_outfile = _args['flight_patterns']
        if not flight_patterns_outfile:
            logger.info('Parameter flight-patterns is required for the flights datatype')
            return

        designated_carriers_outfile = _args['designated_carriers']
        if not designated_carriers_outfile:
            logger.info('Parameter designated-carriers is required for the flights datatype')
            return

        flying_carriers_outfile = _args['flying_carriers']
        if not flying_carriers_outfile:
            logger.info('Parameter flying-carriers is required for the flights datatype')
            return

        codeshares_outfile = _args['codeshares']
        if not codeshares_outfile:
            logger.info('Parameter codeshares is required for the flights datatype')
            return

        logger.info('Parsing flights')
        with zipfile.ZipFile(infile) as file_zip:
            for file_info in file_zip.infolist():
                with file_zip.open(file_info) as file_content:
                    if file_info.filename.lower() != 'ssim.dat':
                        logger.info('Unexpected entry in ssim zip-file: {}.'.format(file_info.filename))
                        continue

                    # store flight bases
                    with open(outfile, 'wt') as flight_bases_file:
                        flights_importer = FlightsImporter(flight_bases_file, logger, text_mode=True)
                        for line_count, line in enumerate(file_content):
                            if line_count and line_count % 100000 == 0:
                                logger.info('Processed {:,} SSIM records'.format(line_count))
                                logger.info(flights_importer.get_flights_count())
                            flights_importer.process(line)
                        flights_importer.flush()

                        # store flight patterns
                        flights_importer.post_process()
                        flight_patterns_count = 0
                        with open(flight_patterns_outfile, 'wt') as flight_patterns_file:
                            for flight_pattern in flights_importer.flight_patterns():
                                flight_patterns_file.write('{}\n'.format(flight_pattern))
                                flight_patterns_count += 1
                        logger.info('Saved to output file {:,} flight patterns'.format(flight_patterns_count))
                        logger.info('Saved to output file {:,} flight bases'.format(flights_importer.flight_bases_count()))

                        # store designated carriers
                        designated_carriers_count = 0
                        with open(designated_carriers_outfile, 'wt') as designated_carriers_file:
                            for designated_carrier in flights_importer.designated_carriers():
                                designated_carriers_file.write('{}\n'.format(designated_carrier))
                                designated_carriers_count += 1
                        logger.info('Saved to output file {:,} designated carriers'.format(designated_carriers_count))

                        # store flying carriers
                        flying_carriers_count = 0
                        with open(flying_carriers_outfile, 'wt') as flying_carriers_file:
                            for flying_carrier in flights_importer.flying_carriers():
                                flying_carriers_file.write('{}\n'.format(flying_carrier))
                                flying_carriers_count += 1
                        logger.info('Saved to output file {:,} flying carriers'.format(flying_carriers_count))

                        # store codeshares
                        codeshares_count = 0
                        with open(codeshares_outfile, 'wt') as codeshares_file:
                            for codeshare in flights_importer.codeshares():
                                codeshares_file.write('{}\n'.format(codeshare))
                                codeshares_count += 1
                        logger.info('Saved to output file {:,} codeshares'.format(codeshares_count))

    if datatype == 'aircraft':
        logger.info('Parsing aircraft')
        aircraft_count = 0
        with open(infile, 'rt') as input_file:
            aircraft_importer = AircraftImporter(logger)
            with open(outfile, 'wb') as result_file:
                for line in input_file:
                    if not line:
                        continue
                    aircraft = aircraft_importer.parse(line)
                    if not aircraft:
                        continue
                    file_util.write_binary_string(result_file, aircraft.SerializeToString())
                    aircraft_count += 1

        logger.info('Saved to file {:,} aircraft'.format(aircraft_count))
        logger.info('Done parsing aircraft.')


if __name__ == '__main__':
    main()
