import six

from datetime import datetime, timedelta

import travel.library.python.dicts.file_util as file_util
import travel.proto.shared_flights.ssim.flights_pb2 as flights_pb2
from travel.avia.shared_flights.lib.python.date_utils.date_index import DateIndex
from travel.avia.shared_flights.lib.python.date_utils.date_matcher import DateMatcher
from travel.avia.shared_flights.lib.python.date_utils.date_shift import get_days_shift
from travel.avia.shared_flights.tasks.ssim_parser.ssim_codeshares import SsimCodesharesMap
from travel.avia.shared_flights.tasks.ssim_parser.ssim_parser_util import SsimParser

RAILWAY_CARRIERS = ['2C', '7T']
END_OF_SCHEDULE = (datetime.now() + timedelta(days=366)).strftime('%Y-%m-%d')

# Specifal value used to mark flight_base buckets where two or more carriers are marked as administrative for the same flight
NO_FLYING_CARRIER = 'X'


# Caches flight pattern that should eventually be linked to the given flight base
class FlightBaseHolder:

    def __init__(self, flight_base, flight_pattern):
        self.flight_base = flight_base
        self.flight_pattern = flight_pattern


# DepartureDateShift can be calculated only within single itinerary variation, hence we need to track it
class ItineraryVariation:

    def __init__(self, carrier, flight_number, leg_number, itinerary_variation):
        self.carrier = carrier
        self.flight_number = flight_number
        self.leg_number = leg_number
        self.itinerary_variation = itinerary_variation

    def is_next_leg(self, other_itinerary_variation):
        return self.carrier == other_itinerary_variation.carrier and \
            self.flight_number == other_itinerary_variation.flight_number and \
            self.itinerary_variation == other_itinerary_variation.itinerary_variation and \
            self.leg_number == other_itinerary_variation.leg_number + 1


class FlightsImporter:
    """
        Parses an SSIM file into a protobuf. For more details regarding the SSIM format,
        see ftp://ns.tais.ru/pub/doc/SSIM-Oct2005.pdf
    """

    _handlers = {
        '010': 'codeshared_legs',
        '050': 'operating_leg',
        '501': 'flight_performance',
        '127': 'designated_carrier',
    }

    def __init__(self, flight_bases_file, logger, text_mode=False):
        self._flight_bases_file = flight_bases_file
        self._logger = logger
        self._flight_patterns = {}
        self._cur_season = ''
        self._current_flight_holder = None
        self._current_flight_pattern = None
        self._current_leg_key = None
        self._current_itinerary_variation = None
        self._current_line_operating_days = 0
        self._text_mode = text_mode
        # maps flight_pattern leg_key to its id
        self._fp_leg_key_to_id = {}
        # maps bucket_key to its operating leg (flight pattern), if any
        self._bucket_key_to_operating_fp = {}
        # maps marketing leg (flight pattern) id to its operating bucket
        self._fp_to_operating_bucket = {}
        # caches marketing legs in case we need to convert them to operating ones
        self._postponed_legs = set()
        # cache for Record4 field 010 data, as we can't process it at the time we read it from SSIM, have to postpone
        self._postponed_010_recs = []
        self._date_index = DateIndex(datetime.now())
        self._date_matcher = DateMatcher(self._date_index)
        self._current_flight_base_id = 0
        self._current_flight_pattern_id = 0
        # maps flight base string (with no id and no ivi) to its id
        self._flight_bases_map = {}
        # maps designated carrier to its ID
        self._designated_carrier_ids = {}
        self._current_designated_carrier_id = 0
        # maps bucket key to the flying carrier (if any)
        self._flying_carriers = {}
        self._codeshares_map = SsimCodesharesMap()

    def flight_bases_count(self):
        return self._current_flight_base_id

    def flight_patterns(self):
        for key, flight_pattern in six.iteritems(self._flight_patterns):
            if not flight_pattern.IsCodeshare or flight_pattern.OperatingFlightPatternId:
                yield flight_pattern

    def designated_carriers(self):
        for dc_title, dc_id in six.iteritems(self._designated_carrier_ids):
            designated_carrier = flights_pb2.TDesignatedCarrier()
            designated_carrier.Id = dc_id
            designated_carrier.Title = dc_title
            yield designated_carrier

    def codeshares(self):
        return self._codeshares_map.protos()

    def flying_carriers(self):
        for bucket, flying_carrier_iata in six.iteritems(self._flying_carriers):
            if flying_carrier_iata == NO_FLYING_CARRIER:
                continue
            flying_carrier = flights_pb2.TFlyingCarrier()
            flying_carrier.Bucket = bucket
            flying_carrier.FlyingCarrierIata = flying_carrier_iata
            yield flying_carrier

    def get_flights_count(self):
        return '{:,} base flights, {:,} flight patterns'.format(self.flight_bases_count(), len(self._flight_patterns))

    def process(self, line):
        if not line:
            return

        record_type = line[0]
        if not record_type == '2' and not record_type == '3' and not record_type == '4':
            return

        if record_type == '2':
            self._cur_season = line[10:12]
            return

        if record_type == '3':
            self._process_record3(line)
            return

        if record_type == '4':
            self._process_record4(line)
            return

    def _process_record3(self, line):
        operational_suffix = line[1]

        if operational_suffix != ' ':
            return

        self.flush_current_pattern_and_base()

        flight_pattern = flights_pb2.TFlightPattern()

        flight_pattern.MarketingCarrierIata = line[2:5].strip()
        flight_pattern.MarketingFlightNumber = line[5:9].strip()
        flight_pattern.IsAdministrative = line[2] == 'A'

        flight_pattern.OperatingFromDate = SsimParser.parse_date(line[14:21], END_OF_SCHEDULE)
        flight_pattern.OperatingUntilDate = SsimParser.parse_date(line[21:28], END_OF_SCHEDULE)

        flight_pattern.OperatingOnDays = SsimParser.parse_week_days_mask(line[28:35])

        if flight_pattern.MarketingCarrierIata in RAILWAY_CARRIERS:
            return

        flight_pattern.Id = self._next_flight_pattern_id()
        line_itinerary_variation = self._flight_params_from_line(line)

        if line_itinerary_variation.leg_number > 1 and line_itinerary_variation.is_next_leg(self._current_itinerary_variation):
            flight_pattern.DepartureDayShift = get_days_shift(self._current_line_operating_days, flight_pattern.OperatingOnDays)

        self._current_itinerary_variation = line_itinerary_variation
        self._current_line_operating_days = flight_pattern.OperatingOnDays

        leg_key = SsimParser.get_leg_key(
            line_itinerary_variation.carrier,
            line_itinerary_variation.flight_number,
            line_itinerary_variation.leg_number,
            line_itinerary_variation.itinerary_variation,
        )
        flight_codeshare_indicator = line[148]
        flight_pattern.FlightLegKey = leg_key
        self._fp_leg_key_to_id[leg_key] = flight_pattern.Id
        flight_pattern.LegSeqNumber = line_itinerary_variation.leg_number

        '''
        Per 8.7.6 chapter from the SSIM manual the value may also be:
        'S': for the 'Airline ABC doing business as XYZ' case;
        'X': for the 'Airline ABC as airline XYZ' case;
        Skipping for now, as it does not change anything (except possibly the brand) in the schedule.
        We'll deal with the brand issue later if needed.
        '''
        if flight_codeshare_indicator not in 'LZ':
            flight_pattern.IsCodeshare = False

            self._current_flight_holder = FlightBaseHolder(self._new_flight_base(line), flight_pattern)
            flight_pattern.BucketKey = SsimParser.get_bucket_key(
                flight_pattern.MarketingCarrierIata,
                flight_pattern.MarketingFlightNumber,
                line_itinerary_variation.leg_number,
            )
            self._append_operating_leg_for_bucket(flight_pattern.BucketKey, flight_pattern.Id)
        else:
            flight_pattern.IsCodeshare = True
            self._postponed_legs.add(flight_pattern.Id)

        self._flight_patterns[flight_pattern.Id] = flight_pattern

    # Record4, field 050
    def operating_leg(self, line):
        leg_key = SsimParser.get_rec4_leg_key(line)

        operating_carrier_iata = line[39:42].strip()
        operating_flight_number = line[42:46].strip()

        if leg_key[3] in RAILWAY_CARRIERS or operating_carrier_iata in RAILWAY_CARRIERS:
            return

        flight_pattern = self._obtain_flight_pattern(leg_key[0])

        if not flight_pattern:
            raise Exception('Leg {} must exist, but it doesn\'t. Line: {}'.format(leg_key, line))

        flight_pattern.BucketKey = SsimParser.get_bucket_key(operating_carrier_iata, operating_flight_number, leg_key[1])
        flight_pattern.IsCodeshare = True
        self._fp_to_operating_bucket[flight_pattern.Id] = flight_pattern.BucketKey
        self._codeshares_map.add_codeshare_050(
            flight_pattern.MarketingCarrierIata,
            flight_pattern.MarketingFlightNumber,
            flight_pattern.LegSeqNumber,
            operating_carrier_iata,
            operating_flight_number,
            flight_pattern.LegSeqNumber,
            flight_pattern.OperatingFromDate,
            flight_pattern.OperatingUntilDate,
        )

    # Record4, field 010
    def codeshared_legs(self, line):
        leg_key_parts = SsimParser.get_rec4_leg_key(line)

        if leg_key_parts[3] in RAILWAY_CARRIERS:
            return

        operating_flight_pattern = self._obtain_flight_pattern(leg_key_parts[0])

        right_pos = 193 if len(line) > 193 else len(line)
        codeshares = line[39:right_pos].split('/')
        for codeshare in codeshares:
            if len(codeshare) < 4:
                continue
            marketing_carrier = codeshare[0:3].strip()
            if marketing_carrier in RAILWAY_CARRIERS:
                continue
            marketing_flight_number = codeshare[3:].strip().lstrip('0')
            marketing_bucket_key = SsimParser.get_bucket_key(marketing_carrier, marketing_flight_number, leg_key_parts[1])
            self._postponed_010_recs.append((marketing_bucket_key, operating_flight_pattern.Id))
            self._codeshares_map.add_codeshare_010(
                marketing_carrier,
                marketing_flight_number,
                operating_flight_pattern.LegSeqNumber,
                operating_flight_pattern.MarketingCarrierIata,
                operating_flight_pattern.MarketingFlightNumber,
                operating_flight_pattern.LegSeqNumber,
                operating_flight_pattern.OperatingFromDate,
                operating_flight_pattern.OperatingUntilDate,
            )

    def flight_performance(self, line):
        leg_key = SsimParser.get_rec4_leg_key(line)
        if leg_key[3] in RAILWAY_CARRIERS:
            return
        flight_pattern = self._obtain_flight_pattern(leg_key[0])

        performance_char = line[39]
        flight_pattern.Performance = 0 if performance_char == 'N' else int(performance_char)

    def designated_carrier(self, line):
        leg_key = SsimParser.get_rec4_leg_key(line)
        if leg_key[3] in RAILWAY_CARRIERS:
            return
        flight_pattern = self._obtain_flight_pattern(leg_key[0])

        right_pos = 193 if len(line) > 193 else len(line)
        designated_carrier = line[39:right_pos].strip().lstrip('/')
        if designated_carrier:
            designated_carrier_id = self._designated_carrier_ids.get(designated_carrier)
            if not designated_carrier_id:
                self._current_designated_carrier_id += 1
                self._designated_carrier_ids[designated_carrier] = self._current_designated_carrier_id
                designated_carrier_id = self._current_designated_carrier_id

            flight_pattern.DesignatedCarrier = designated_carrier_id
            if self._current_flight_holder and not flight_pattern.IsCodeshare:
                self._current_flight_holder.flight_base.DesignatedCarrier = designated_carrier_id

    def flush_current_pattern_and_base(self):
        if self._current_flight_holder:
            current_flight_base = self._current_flight_holder.flight_base
            flight_str = str(current_flight_base)
            existing_flight_id = self._flight_bases_map.get(flight_str)
            if existing_flight_id:
                current_flight_base.Id = existing_flight_id
            else:
                self._current_flight_base_id += 1
                current_flight_base.Id = self._current_flight_base_id
                self._flight_bases_map[flight_str] = current_flight_base.Id

            current_flight_base.ItineraryVariationIdentifier = str(current_flight_base.Id)
            flight_base_bucketKey = SsimParser.get_bucket_key(
                current_flight_base.OperatingCarrierIata,
                current_flight_base.OperatingFlightNumber,
                current_flight_base.LegSeqNumber,
            )
            if not existing_flight_id:
                flying_carrier = current_flight_base.FlyingCarrierIata
                # Some carriers put an 'X' mark into this field, not sure why, let's ignore it as it is not a valid carrier
                if flying_carrier and flying_carrier != current_flight_base.OperatingCarrierIata and flying_carrier != NO_FLYING_CARRIER:
                    existing_flying_carrier = self._flying_carriers.get(flight_base_bucketKey)
                    # Use 'X' to mark the case we're unable to support at this time:
                    # when the same flight is operated by different carriers on different dates
                    if existing_flying_carrier and existing_flying_carrier != 'X' and existing_flying_carrier != flying_carrier:
                        self._flying_carriers[flight_base_bucketKey] = NO_FLYING_CARRIER
                    if not existing_flying_carrier:
                        self._flying_carriers[flight_base_bucketKey] = flying_carrier
                if self._text_mode:
                    self._flight_bases_file.write('{}\n'.format(current_flight_base))
                else:
                    file_util.write_binary_string(self._flight_bases_file, current_flight_base.SerializeToString())

            if self._current_flight_holder:
                self._current_flight_holder.flight_pattern.FlightId = current_flight_base.Id

            self._current_flight_holder = None
        self.flush()

    def flush(self):
        if self._current_flight_pattern and self._current_leg_key:
            self._flight_patterns[self._current_flight_pattern.Id] = self._current_flight_pattern

        self._current_flight_pattern = None
        self._current_leg_key = None

    def post_process(self):
        self.flush_current_pattern_and_base()

        # Print out intermediate results for the iata correction tables - to make it easier to figure out if something already went wrong
        self._logger.info('Parsed designated carriers: %s', len(self._designated_carrier_ids))
        self._logger.info('Parsed flying carriers: %s', len(list(self.flying_carriers())))

        # process postponed records from Record4, field 050
        # scan 1
        self._logger.info('Start flight patterns post-processing, scan 1 (records to process: %s)', len(self._fp_to_operating_bucket))
        flight_patterns_for_scan2 = {}
        for flight_pattern_id, bucket_key in six.iteritems(self._fp_to_operating_bucket):
            operating_pattern_ids = self._bucket_key_to_operating_fp.get(bucket_key)
            matching_operating_legs = self._dates_overlap(operating_pattern_ids, flight_pattern_id)
            if not matching_operating_legs:
                flight_patterns_for_scan2[flight_pattern_id] = bucket_key
                continue

            index = 0
            flight_pattern = self._flight_patterns[flight_pattern_id]
            self._remove_leg_from_postponed(flight_pattern_id)
            marketing_bucket_key = SsimParser.get_bucket_key(
                flight_pattern.MarketingCarrierIata,
                flight_pattern.MarketingFlightNumber,
                flight_pattern.LegSeqNumber,
            )
            for operating_pattern_id, overlap in six.iteritems(matching_operating_legs):
                operating_flight_pattern = self._flight_patterns[operating_pattern_id]
                # To avoid removing elems from huge dictionary too often, and thus speed things up
                new_flight_pattern = flights_pb2.TFlightPattern()
                new_flight_pattern.MergeFrom(flight_pattern)
                new_flight_pattern.FlightId = operating_flight_pattern.FlightId
                new_flight_pattern.BucketKey = operating_flight_pattern.BucketKey
                new_flight_pattern.OperatingFromDate = overlap[0]
                new_flight_pattern.OperatingUntilDate = overlap[1]
                new_flight_pattern.OperatingOnDays = overlap[2]
                new_flight_pattern.OperatingFlightPatternId = operating_flight_pattern.Id
                if index == 0:
                    new_flight_pattern.Id = flight_pattern.Id
                else:
                    new_flight_pattern.Id = self._next_flight_pattern_id()
                index += 1

                self._flight_patterns[new_flight_pattern.Id] = new_flight_pattern
                self._append_operating_leg_for_bucket(marketing_bucket_key, operating_pattern_id)

        # scan 2
        self._logger.info('Start flight patterns post-processing, scan 2 (records to process: %s)', len(flight_patterns_for_scan2))
        for flight_pattern_id, bucket_key in six.iteritems(flight_patterns_for_scan2):
            flight_pattern = self._flight_patterns[flight_pattern_id]
            operating_pattern_ids = self._bucket_key_to_operating_fp.get(bucket_key)
            matching_operating_legs = self._dates_overlap(operating_pattern_ids, flight_pattern_id)
            if not matching_operating_legs:
                if flight_pattern.OperatingFromDate < END_OF_SCHEDULE:
                    self._logger.warn(
                        'SSIM Error: unable to resolve codeshare reference for leg %s',
                        self._flight_patterns[flight_pattern_id]
                    )
                continue

            index = 0
            self._remove_leg_from_postponed(flight_pattern_id)
            marketing_bucket_key = SsimParser.get_bucket_key(
                flight_pattern.MarketingCarrierIata,
                flight_pattern.MarketingFlightNumber,
                flight_pattern.LegSeqNumber,
            )
            for operating_pattern_id, overlap in six.iteritems(matching_operating_legs):
                operating_flight_pattern = self._flight_patterns[operating_pattern_id]
                # To avoid removing elems from huge dictionary too often, and thus speed things up
                new_flight_pattern = flights_pb2.TFlightPattern()
                new_flight_pattern.MergeFrom(flight_pattern)
                new_flight_pattern.FlightId = operating_flight_pattern.FlightId
                new_flight_pattern.BucketKey = operating_flight_pattern.BucketKey
                new_flight_pattern.OperatingFromDate = overlap[0]
                new_flight_pattern.OperatingUntilDate = overlap[1]
                new_flight_pattern.OperatingOnDays = overlap[2]
                new_flight_pattern.OperatingFlightPatternId = operating_flight_pattern.Id
                if index == 0:
                    new_flight_pattern.Id = flight_pattern.Id
                else:
                    new_flight_pattern.Id = self._next_flight_pattern_id()
                index += 1

                self._flight_patterns[new_flight_pattern.Id] = new_flight_pattern
                self._append_operating_leg_for_bucket(marketing_bucket_key, operating_pattern_id)

        # process postponed records from Record4, field 010
        self._logger.info('Start flight patterns post-processing, scan 010 (records to process: %s)', len(self._postponed_010_recs))
        for record in self._postponed_010_recs:
            marketing_bucket_key = record[0]
            operating_flight_pattern_id = record[1]
            original_operating_leg = self._flight_patterns[operating_flight_pattern_id]
            operating_pattern_ids = self._bucket_key_to_operating_fp.get(marketing_bucket_key)
            filtered_operating_pattern_ids = []
            if operating_pattern_ids:
                for leg_id in operating_pattern_ids:
                    flight_pattern = self._flight_patterns[leg_id]
                    same_carrier = flight_pattern.MarketingCarrierIata == original_operating_leg.MarketingCarrierIata
                    same_flight = flight_pattern.MarketingFlightNumber == original_operating_leg.MarketingFlightNumber
                    same_leg = flight_pattern.LegSeqNumber == original_operating_leg.LegSeqNumber
                    if same_carrier and same_flight and same_leg:
                        filtered_operating_pattern_ids.append(leg_id)

            matching_operating_legs = self._dates_overlap(filtered_operating_pattern_ids, operating_flight_pattern_id)

            # Found at least something that operates with this bucket key - let's not override that data from the Rec4 050 field
            if matching_operating_legs:
                continue

            # Found new codeshare reference in Rec4 010 field - add it to the flight patterns map
            new_flight_pattern = flights_pb2.TFlightPattern()
            new_flight_pattern.MergeFrom(self._flight_patterns[operating_flight_pattern_id])
            new_flight_pattern.IsCodeshare = True
            new_flight_pattern.Id = self._next_flight_pattern_id()
            marketing_bucket_key_parts = marketing_bucket_key.split('.')
            new_flight_pattern.MarketingCarrierIata = marketing_bucket_key_parts[0]
            new_flight_pattern.MarketingFlightNumber = marketing_bucket_key_parts[1]
            new_flight_pattern.OperatingFlightPatternId = operating_flight_pattern_id
            new_leg_key = SsimParser.get_leg_key(
                marketing_bucket_key_parts[0],
                marketing_bucket_key_parts[1],
                new_flight_pattern.LegSeqNumber,
                new_flight_pattern.Id,
            )
            new_flight_pattern.FlightLegKey = new_leg_key
            self._flight_patterns[new_flight_pattern.Id] = new_flight_pattern

        # verify that we have no leftovers in non-processed postponed records
        if self._postponed_legs:
            # Skip legs that only operate 'forcibly' on the last day - it's normal that there's no match
            unprocessed_count = 0
            for leg_id in self._postponed_legs:
                flight_pattern = self._flight_patterns[leg_id]
                # Remove unprocess flight pattern from the resulting list
                self._flight_patterns.pop(leg_id, None)
                if flight_pattern.OperatingFromDate >= END_OF_SCHEDULE:
                    continue
                if unprocessed_count < 100:
                    self._logger.warn('Unprocessed leg: %s', flight_pattern)
                unprocessed_count += 1

            if unprocessed_count:
                self._logger.warn('SSIM Error: %d legs were not processed', unprocessed_count)

    def _flight_params_from_line(self, line):
        line_carrier_iata = line[2:5].strip()
        line_flight_number = line[5:9].strip()
        leg_seq_number = int('1' + line[11:13]) - 100
        itinerary_variation_identifier = '{}{}'.format(line[127], line[9:11])
        return ItineraryVariation(line_carrier_iata, line_flight_number, leg_seq_number, itinerary_variation_identifier)

    def _new_flight_base(self, line):
        # type: (str) -> (flights_pb2.TFlightBase, bool)
        line_itinerary_variation = self._flight_params_from_line(line)

        flight = flights_pb2.TFlightBase()
        flight.OperatingCarrierIata = line_itinerary_variation.carrier
        flight.OperatingFlightNumber = line_itinerary_variation.flight_number
        flight.LegSeqNumber = line_itinerary_variation.leg_number
        flight.ServiceType = line[13]

        flight.DepartureStationIata = line[36:39]
        flight.ScheduledDepartureTime = SsimParser.parse_time(line[39:43])
        flight.DepartureTerminal = line[52:54].strip()

        flight.ArrivalStationIata = line[54:57]
        flight.ScheduledArrivalTime = SsimParser.parse_time(line[61:65])
        flight.ArrivalTerminal = line[70:72].strip()
        flight.AircraftModel = line[72:75]
        flight.IntlDomesticStatus = line[119:121]
        flight.Season = self._cur_season

        flying_carrier_iata = line[128:131].strip()
        if flying_carrier_iata:
            flight.FlyingCarrierIata = flying_carrier_iata

        return flight

    def _process_record4(self, line):
        operational_suffix = line[1]

        if operational_suffix != ' ':
            return

        data_element_identifier = line[30:33]
        handler = self._handlers.get(data_element_identifier)
        if handler:
            getattr(self, handler)(line)

    def _obtain_flight_pattern(self, leg_key):
        # type: (str) -> flights_pb2.TFlightPattern
        if leg_key == self._current_leg_key:
            return self._current_flight_pattern

        self.flush()

        flight_pattern_id = self._fp_leg_key_to_id.get(leg_key)
        if not flight_pattern_id:
            return None

        flight_pattern = self._flight_patterns.get(flight_pattern_id)
        if not flight_pattern:
            return None

        self._current_leg_key = leg_key
        self._current_flight_pattern = flight_pattern
        return flight_pattern

    def _append_operating_leg_for_bucket(self, bucket_key, flight_pattern_id):
        existing_values = self._bucket_key_to_operating_fp.get(bucket_key)
        if existing_values:
            if flight_pattern_id not in existing_values:
                existing_values.append(flight_pattern_id)
        else:
            self._bucket_key_to_operating_fp[bucket_key] = [flight_pattern_id]

    def _next_flight_pattern_id(self):
        self._current_flight_pattern_id += 1
        return self._current_flight_pattern_id

    def _dates_overlap(self, operating_pattern_ids, flight_pattern_id):
        result = {}
        if not operating_pattern_ids:
            return result

        leg1 = self._flight_patterns[flight_pattern_id]
        for leg_id in operating_pattern_ids:
            leg2 = self._flight_patterns[leg_id]

            overlap = self._date_matcher.intersect(
                leg1.OperatingFromDate,
                leg1.OperatingUntilDate,
                leg1.OperatingOnDays,
                leg2.OperatingFromDate,
                leg2.OperatingUntilDate,
                leg2.OperatingOnDays,
            )
            if not overlap:
                continue
            result[leg_id] = overlap
        return result

    def _remove_leg_from_postponed(self, flight_pattern_id):
        if flight_pattern_id in self._postponed_legs:
            self._postponed_legs.remove(flight_pattern_id)

    def flight_patterns_by_leg_key(self):
        result = {}
        for _, flight_pattern in six.iteritems(self._flight_patterns):
            result[flight_pattern.FlightLegKey] = flight_pattern
        return result

    # For the sake of testing convenience
    def postponed_legs(self):
        return [self._flight_patterns[leg] for leg in self._postponed_legs]
