# coding=utf-8

from datetime import datetime
from google.protobuf import json_format
import tempfile
import zipfile

import paramiko

from travel.avia.library.python.shared_dicts.rasp import get_repository, ResourceType
from travel.avia.shared_flights.tasks.ssim_parser.aircraft import AircraftImporter
from travel.avia.shared_flights.tasks.ssim_parser.flights import FlightsImporter
from travel.library.python.dicts import file_util
from travel.proto.dicts.rasp.transport_model_pb2 import TTransportModel
from travel.proto.dicts.rasp.transport_pb2 import TTransport


class SsimFetchResult(object):
    """ Result of SSIM files fetching. """

    def __init__(self, current_ssim_date, aircraft_output_file, flight_bases_file, flight_patterns_file,
                 designated_carriers_file, flying_carriers_file, codeshares_file, new_transport_models):
        self._current_ssim_date = current_ssim_date
        self._aircraft_output_file = aircraft_output_file
        self._flight_bases_file = flight_bases_file
        self._flight_patterns_file = flight_patterns_file
        self._designated_carriers_file = designated_carriers_file
        self._flying_carriers_file = flying_carriers_file
        self._codeshares_file = codeshares_file
        self._new_transport_models = new_transport_models

    @property
    def current_ssim_date(self):
        return self._current_ssim_date

    @property
    def aircraft_output_file(self):
        return self._aircraft_output_file

    @property
    def flight_bases_file(self):
        return self._flight_bases_file

    @property
    def flight_patterns_file(self):
        return self._flight_patterns_file

    @property
    def designated_carriers_file(self):
        return self._designated_carriers_file

    @property
    def flying_carriers_file(self):
        return self._flying_carriers_file

    @property
    def codeshares_file(self):
        return self._codeshares_file

    @property
    def new_transport_models(self):
        return self._new_transport_models


class SsimFetcher(object):
    """ Fetches SSIM files over sftp. """

    def __init__(self, hostname, username, password, logger):
        self._hostname = hostname
        self._username = username
        self._password = password
        self._logger = logger

    def fetch(self, last_parsed_date, parse_aircrafts_only=False, oauth_token=None):
        with paramiko.SSHClient() as ssh_client:
            ssh_client.set_missing_host_key_policy(paramiko.AutoAddPolicy())

            ssh_client.connect(
                hostname=self._hostname,
                username=self._username,
                password=self._password,
                look_for_keys=False,
                allow_agent=False,
                timeout=2700,
            )

            transport = ssh_client.get_transport()
            transport.default_window_size = paramiko.common.MAX_WINDOW_SIZE
            transport.default_max_packet_size = paramiko.common.MIN_PACKET_SIZE
            ssim_file_name = 'ssim.zip'
            with ssh_client.open_sftp() as sftp:
                current_ssim_date = None
                for sftp_attr in sftp.listdir_attr():
                    if sftp_attr.filename == ssim_file_name and sftp_attr.st_mtime:
                        current_ssim_date = datetime.utcfromtimestamp(sftp_attr.st_mtime).strftime('%Y-%m-%d %H:%M:%S')

                if not current_ssim_date:
                    raise Exception(
                        'Unable to locate {} on {}. Please update the task parameters.'.format(
                            ssim_file_name,
                            self._ftp_host,
                        )
                    )

                self._logger.info('The latest SSIM files found are from {}'.format(current_ssim_date))

                if last_parsed_date is not None and last_parsed_date >= current_ssim_date:
                    self._logger.info('Last parsed date {} is already no older than the current one {}'.format(
                        last_parsed_date, current_ssim_date))
                    return None

                # parse aircraft
                aircraft_input_file = tempfile.NamedTemporaryFile().name
                aircraft_output_file = tempfile.NamedTemporaryFile().name
                sftp.get('equipment.dat', aircraft_input_file)
                aircrafts = self.parse_aircraft(aircraft_input_file, aircraft_output_file)
                self._logger.info('Temporary stored aircraft at {}'.format(aircraft_output_file))

                # parse flights
                if not parse_aircrafts_only:
                    ssim_file = tempfile.NamedTemporaryFile().name
                    sftp.get(ssim_file_name, ssim_file)

                    flight_bases_file = tempfile.NamedTemporaryFile().name
                    flight_patterns_file = tempfile.NamedTemporaryFile().name
                    designated_carriers_file = tempfile.NamedTemporaryFile().name
                    flying_carriers_file = tempfile.NamedTemporaryFile().name
                    codeshares_file = tempfile.NamedTemporaryFile().name

                    self.parse_flights(
                        ssim_file,
                        flight_bases_file,
                        flight_patterns_file,
                        designated_carriers_file,
                        flying_carriers_file,
                        codeshares_file,
                    )
                    self._logger.info(
                        'Temporary stored flights at {}, {}'.format(flight_bases_file, flight_patterns_file))
                    self._logger.info('Temporary stored designated carriers at %s', designated_carriers_file)
                    self._logger.info('Temporary stored flying carriers at %s', flying_carriers_file)
                else:
                    flight_bases_file = None
                    flight_patterns_file = None
                    designated_carriers_file = None
                    flying_carriers_file = None
                    codeshares_file = None
                    self._logger.info('Skipped parsing flights data')

        new_transport_models = self.compare_transport_models(aircrafts, oauth_token)

        return SsimFetchResult(
            current_ssim_date,
            aircraft_output_file,
            flight_bases_file,
            flight_patterns_file,
            designated_carriers_file,
            flying_carriers_file,
            codeshares_file,
            new_transport_models,
        )

    def parse_aircraft(self, input_file, output_file):
        self._logger.info('Parsing aircraft')
        aircraft_count = 0
        aircrafts = []
        with open(input_file, 'rt') as input:
            aircraft_importer = AircraftImporter(self._logger)
            with open(output_file, 'wb') as output:
                for line in input:
                    if not line:
                        continue
                    aircraft = aircraft_importer.parse(line)
                    if not aircraft:
                        continue
                    aircrafts.append(aircraft)
                    file_util.write_binary_string(output, aircraft.SerializeToString())
                    aircraft_count += 1

        self._logger.info('Saved to temp file {:,} aircraft'.format(aircraft_count))
        self._logger.info('Done parsing aircraft')
        return aircrafts

    def compare_transport_models(self, aircrafts, oauth_token):
        new_transport_models = []
        self._logger.info('Looking for the new transport models')
        transport_models_repository = get_repository(
            ResourceType.TRAVEL_DICT_RASP_TRANSPORT_MODEL_PROD,
            oauth=oauth_token,
        )
        transport_models = {}
        for transport_model in transport_models_repository.itervalues():
            transport_type = transport_model.TransportType
            if transport_type != TTransport.EType.TYPE_PLANE and transport_type != TTransport.EType.TYPE_HELICOPTER:
                continue
            transport_models[transport_model.Code] = transport_model
            transport_models[transport_model.CodeEn] = transport_model

        for aircraft in aircrafts:
            transport_model = transport_models.get(aircraft.SubModel)
            if not transport_model:
                # found new transport model
                new_transport_model = TTransportModel()
                new_transport_model.CodeEn = aircraft.SubModel
                new_transport_model.TitleEn = aircraft.Title
                new_transport_model.PropellerFlight = 'Jet' not in aircraft.Type
                new_transport_model.PlaneBodyType = 'wire' if aircraft.IsWideBody else 'standard'
                new_transport_models.append(json_format.MessageToJson(new_transport_model))

        self._logger.info('Done looking for the new transport models')
        return sorted(new_transport_models)

    def parse_flights(self, ssim_file, flight_bases_file, flight_patterns_file, designated_carriers_file,
                      flying_carriers_file, codeshares_file):
        self._logger.info('Parsing flights')
        with zipfile.ZipFile(ssim_file) as file_zip:
            for file_info in file_zip.infolist():
                with file_zip.open(file_info) as file_content:
                    if file_info.filename.lower() != 'ssim.dat':
                        raise Exception('Unexpected entry in ssim zip-file: {}.'.format(file_info.filename))
                    self._process_flights_file_content(
                        file_content,
                        flight_bases_file,
                        flight_patterns_file,
                        designated_carriers_file,
                        flying_carriers_file,
                        codeshares_file,
                    )

        self._logger.info('Done parsing flights')

    def _process_flights_file_content(self, file_content, flight_bases_file, flight_patterns_file,
                                      designated_carriers_file, flying_carriers_file, codeshares_file):
        with open(flight_bases_file, 'wb') as flight_bases:
            flights_importer = FlightsImporter(flight_bases, self._logger)
            line_count = 0
            for line in file_content:
                line_count += 1
                if line_count % 100000 == 0:
                    self._logger.info('Processed {:,} SSIM records'.format(line_count))
                    self._logger.info(flights_importer.get_flights_count())
                flights_importer.process(line)
            flights_importer.flush()

            # store flight patterns
            flights_importer.post_process()
            flight_patterns_count = 0
            with open(flight_patterns_file, 'wb') as flight_patterns:
                for flight_pattern in flights_importer.flight_patterns():
                    flight_patterns_count += 1
                    file_util.write_binary_string(flight_patterns, flight_pattern.SerializeToString())
            self._logger.info('Saved to temp file {:,} flight bases'.format(flights_importer.flight_bases_count()))
            self._logger.info('Saved to temp file {:,} flight patterns'.format(flight_patterns_count))

            # store designated carriers
            designated_carriers_count = 0
            with open(designated_carriers_file, 'wb') as designated_carriers:
                for designated_carrier in flights_importer.designated_carriers():
                    designated_carriers_count += 1
                    file_util.write_binary_string(designated_carriers, designated_carrier.SerializeToString())

            # store flying carriers
            flying_carriers_count = 0
            with open(flying_carriers_file, 'wb') as flying_carriers:
                for flying_carrier in flights_importer.flying_carriers():
                    flying_carriers_count += 1
                    file_util.write_binary_string(flying_carriers, flying_carrier.SerializeToString())

            # store codeshares
            codeshares_count = 0
            with open(codeshares_file, 'wb') as codeshares:
                for codeshare in flights_importer.codeshares():
                    codeshares_count += 1
                    file_util.write_binary_string(codeshares, codeshare.SerializeToString())
            self._logger.info('Saved to temp file {:,} codeshares'.format(codeshares_count))
