#  coding=utf-8
from __future__ import unicode_literals

import datetime


class SsimParser(object):
    """ OAG/SSIM format parsing helper methods """

    _SSIM_MONTHS = {
        'JAN': 1,
        'FEB': 2,
        'MAR': 3,
        'APR': 4,
        'MAY': 5,
        'JUN': 6,
        'JUL': 7,
        'AUG': 8,
        'SEP': 9,
        'OCT': 10,
        'NOV': 11,
        'DEC': 12
    }

    @staticmethod
    def parse_date(data, maxDateValue):
        # type: (str) -> str
        day = int(data[:2])
        year = int(data[-2:]) + 2000
        month_text = data[2:-2]
        month = SsimParser._SSIM_MONTHS.get(month_text)

        if month_text == 'XXX':
            return maxDateValue
        else:
            result = datetime.date(year, month, day)

        str_result = result.strftime('%Y-%m-%d')
        if str_result > maxDateValue:
            return maxDateValue

        return str_result

    @staticmethod
    def parse_time(time_text):
        # type: (str) -> int
        time_number = int('1' + time_text)
        return time_number-10000

    @staticmethod
    def parse_time_zone(tz_text):
        # type: (str) -> int
        sign = -1 if tz_text[1] == '-' else 1
        hours = int(tz_text[1:3])
        minutes = int(tz_text[3:])
        return sign*(hours * 60 + minutes)

    @staticmethod
    def parse_week_days_mask(mask_text):
        # type: (str) -> int
        result = 0
        for c in mask_text:
            if not c or c == ' ':
                continue
            result = result * 10 + int(c)
        return result

    @staticmethod
    def get_leg_key(marketing_carrier_iata, marketing_flight_number, leg_seq_number, itinerary_variation_identifier):
        return '{}.{}.{}.{}'.format(
            marketing_carrier_iata,
            marketing_flight_number.lstrip(' ').lstrip('0'),
            leg_seq_number,
            itinerary_variation_identifier,
        )

    @staticmethod
    def get_bucket_key(operating_carrier_iata, operating_flight_number, leg_seq_number):
        return '{}.{}.{}'.format(
            operating_carrier_iata,
            operating_flight_number.lstrip(' ').lstrip('0'),
            leg_seq_number,
        )

    @staticmethod
    def get_rec4_leg_key(line, leg_seq_number=-1):
        # type (str, int) -> (str, int, str)
        marketing_carrier_iata = line[2:5].strip()
        marketing_flight_number = line[5:9].strip()
        if leg_seq_number < 0:
            leg_seq_number = int('1' + line[11:13]) - 100
        itinerary_variation_identifier = '{}{}'.format(line[27], line[9:11])
        return (
            SsimParser.get_leg_key(
                marketing_carrier_iata,
                marketing_flight_number,
                leg_seq_number,
                itinerary_variation_identifier,
            ),
            leg_seq_number,
            itinerary_variation_identifier,
            marketing_carrier_iata,
        )

    @staticmethod
    def dates_overlap(fp_opertaing_from, fp_operating_until, variation_from, variation_until):
        if fp_opertaing_from > variation_until or fp_operating_until < variation_from:
            return (None, None, True)

        from_date = fp_opertaing_from if fp_opertaing_from >= variation_from else variation_from
        until_date = fp_operating_until if fp_operating_until <= variation_until else variation_until
        return (from_date, until_date, False)
