package main

import (
	"context"
	"fmt"
	"strconv"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/cachedtvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/avia/sirena-proxy/internal/app"
	sirenahandler "a.yandex-team.ru/travel/avia/sirena-proxy/internal/sirena/handler"
	"a.yandex-team.ru/travel/library/go/configuration"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
)

const serviceName = "sirena-http-proxy"

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &app.Cfg)
	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.New(&app.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}
	logger.Info("Loaded configuration", log.Reflect("config", app.Cfg))

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
		ctxCancel()
	}()

	var tvmClient tvm.Client
	tvmAllowedIds := tvmutil.TvmClientIDFromInt(app.Cfg.Tvm.WhiteList)
	if app.Cfg.Tvm.Enabled {
		tvmClient, err = tvmtool.NewDeployClient(tvmtool.WithSrc(strconv.Itoa(app.Cfg.Tvm.SelfAppID)))
		if err != nil {
			logger.Fatal("failed to create tvm client", log.Error(err))
		}
		// At 100 rps we had 0.1 rps errors (exceeded timeout 0.5 seconds) while check service-ticket
		// in tvm-daemon in deploy using standard client. See RASPTICKETS-20639 for details
		// So use cached-tvm-client
		// You can set cache options via cachedtvm.WithCheckServiceTicket and cachedtvm.WithCheckUserTicket
		tvmClient, err = cachedtvm.NewClient(
			tvmClient,
			cachedtvm.WithCheckServiceTicket(
				app.Cfg.Tvm.CacheTTL,
				app.Cfg.Tvm.CacheMaxItems,
			),
		)
		if err != nil {
			logger.Fatal("failed to create cached tvm client", log.Error(err))
		}
	}

	rootRegistry := metrics.NewRegistryWithDeployTags()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, app.Cfg.Metrics.PerfMetricsRefreshInterval)

	// bootstrapping business logic
	httpSirenaHandler := sirenahandler.NewHTTPSirenaHandler(logger, &app.Cfg.Sirena)

	// Metrics server
	go func() {
		err = metricserver.RunMetricsHTTPServer(context.Background(), app.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	// add readiness and shutdown hooks
	probeState := probes.NewState(logger)
	probeRouteBuilders := probes.GetChiRouteBuilders(&app.Cfg.Probes, probeState)
	httpRouteBuilders := []httpserver.RouteBuilder{
		httpSirenaHandler.GetRouteBuilder(),
	}

	// run server
	wg := sync.WaitGroup{}
	wg.Add(1)

	go func() {
		defer wg.Done()
		server := httpserver.NewHTTPServerBuilder(app.Cfg.HTTP, logger, httpRouteBuilders, rootRegistry.WithPrefix("http")).
			WithLoggingRecoverer().
			WithTVM(tvmClient, tvmAllowedIds).
			WithUnprotectedRoutes(probeRouteBuilders).
			Build()
		err = server.Run(context.Background())
		if err != nil {
			logger.Fatal("Error while starting HTTP server", log.Error(err))
		}
	}()

	logger.Info("Started")
	wg.Wait()
}
