package handler

import (
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"

	"github.com/go-chi/chi/v5"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/sirena-proxy/internal/app"
	"a.yandex-team.ru/travel/avia/sirena-proxy/internal/sirena"
)

type HTTPSirenaHandler struct {
	logger log.Logger
	Sirena *app.SirenaConfig
}

type response struct {
	Response string `json:"response"`
}

func NewHTTPSirenaHandler(l log.Logger, Sirena *app.SirenaConfig) *HTTPSirenaHandler {
	return &HTTPSirenaHandler{logger: l, Sirena: Sirena}
}

/* Example request

POST
http://localhost:80/v1/sirena_api?host=localhost&port=34321&client_id=2979
*/

const PortAliasField = "port_alias"
const PortField = "port"
const clientIDField = "client_id"
const clientIDAliasField = "client_code"

func (h *HTTPSirenaHandler) writeError(rw http.ResponseWriter, code int, message string, values ...interface{}) {
	rw.WriteHeader(code)
	if len(values) > 0 {
		message = fmt.Sprintf(message, values...)
	}
	_, _ = rw.Write([]byte(message))
}

func (h *HTTPSirenaHandler) Handle(rw http.ResponseWriter, r *http.Request) {
	span, _ := opentracing.StartSpanFromContext(r.Context(), "internal.sirena.handler.HTTPSirenaHandler: Handle")
	defer span.Finish()

	q := r.URL.Query()

	sirenaConfig := sirena.ConnectionConfig{
		Host: h.Sirena.Host,
	}
	if !h.fillSirenaPortFromParams(rw, q, &sirenaConfig) {
		return
	}

	if !h.fillSirenaClientIDFromParams(rw, q, &sirenaConfig) {
		return
	}

	body, err := io.ReadAll(r.Body)
	defer func(Body io.ReadCloser) {
		_ = Body.Close()
	}(r.Body)
	if err != nil {
		h.writeError(rw, http.StatusInternalServerError, err.Error())
		return
	}
	h.logger.Infof("Read request from POST message of size %d byte", len(body))

	data, err := sirena.ServeSirenaRequest(h.logger, r.Context(), sirenaConfig, body)
	if err != nil {
		h.logger.Error("ServeSirenaRequest failed", log.Error(err))
		h.writeError(rw, http.StatusInternalServerError, err.Error())
		return
	}

	rw.WriteHeader(200)
	written, err := rw.Write(data)
	if err != nil {
		h.writeError(rw, http.StatusInternalServerError, err.Error())
		return
	}

	h.logger.Infof("Read %d bytes from Sirena\n", written)
}

func (h *HTTPSirenaHandler) fillSirenaClientIDFromParams(rw http.ResponseWriter, q url.Values, sirenaConfig *sirena.ConnectionConfig) bool {
	if q.Has(clientIDField) && q.Has(clientIDAliasField) {
		h.writeError(rw, http.StatusBadRequest, "%s or %s fields should be provided, not both", clientIDField, clientIDAliasField)
		return false
	} else if q.Has(clientIDField) && !q.Has(clientIDAliasField) {
		clientID := q.Get(clientIDField)
		IntclientID, err := strconv.Atoi(clientID)
		if err != nil {
			h.writeError(rw, http.StatusBadRequest, "%s: invalid value: %v", clientIDField, err.Error())
			return false
		}
		sirenaConfig.ClientID = uint16(IntclientID)
	} else if !q.Has(clientIDField) && q.Has(clientIDAliasField) {
		alias := q.Get(clientIDAliasField)
		var aliasIsValid bool
		if sirenaConfig.ClientID, aliasIsValid = h.Sirena.ClientAlias[alias]; !aliasIsValid {
			h.writeError(
				rw,
				http.StatusBadRequest,
				"`%s` does not match any known clients: %v. "+
					"Check spelling or use `%s` field instead",
				alias, h.Sirena.ClientAlias, clientIDField,
			)
			return false
		}
	} else if !q.Has(clientIDField) && !q.Has(clientIDAliasField) {
		h.writeError(rw, http.StatusBadRequest, "%s or %s fields should be provided", clientIDField, clientIDAliasField)
		return false
	}
	return true
}

func (h *HTTPSirenaHandler) fillSirenaPortFromParams(rw http.ResponseWriter, q url.Values, sirenaConfig *sirena.ConnectionConfig) bool {
	if q.Has(PortField) && q.Has(PortAliasField) {
		h.writeError(rw, http.StatusBadRequest, "%s or %s fields should be provided, not both", PortField, PortAliasField)
		return false
	} else if q.Has(PortField) && !q.Has(PortAliasField) {
		sirenaConfig.Port = q.Get(PortField)
	} else if !q.Has(PortField) && q.Has(PortAliasField) {
		alias := q.Get(PortAliasField)
		var aliasIsValid bool
		if sirenaConfig.Port, aliasIsValid = h.Sirena.PortAlias[alias]; !aliasIsValid {
			h.writeError(
				rw,
				http.StatusBadRequest,
				"`%s` doesnt match any known alias: %v. "+
					"Check spelling or use `%s` field instead",
				alias, h.Sirena.PortAlias, PortField,
			)
			return false
		}
	} else if !q.Has(PortField) && !q.Has(PortAliasField) {
		h.writeError(rw, http.StatusBadRequest, "%s or %s fields should be provided", PortField, PortAliasField)
		return false
	}
	return true
}

func (h *HTTPSirenaHandler) GetRouteBuilder() func(r chi.Router) {
	return func(r chi.Router) {
		r.Post("/v1/sirena_api", h.Handle)
	}
}
