# -*- coding: utf-8 -*-
from __future__ import division

import calendar
from copy import copy
from collections import defaultdict, namedtuple, OrderedDict
from datetime import datetime, timedelta

from dateutil.relativedelta import relativedelta
from django import forms
from django.conf import settings
from django.conf.urls import url
from django.contrib import admin
from django.contrib.admin import widgets
from django.contrib.admin.filters import SimpleListFilter
from django.contrib.contenttypes.models import ContentType
from django.db.models import Count
from django.template.response import TemplateResponse
from django.utils.http import urlencode

from travel.avia.stat_admin.data.models import PARTNER_REVIEW_RESULT_CHOICES, PartnerReview, ScreenShot
from travel.avia.stat_admin.data.admin.overrides import AviaDateRangeFilter
from travel.avia.stat_admin.lib.review_filter import partner_review_filter
from travel.avia.stat_admin.lib.point_key import get_point_by_point_key


class ScreenshotsInline(admin.StackedInline):
    model = ScreenShot
    fields = ('url_link', 'img')
    readonly_fields = ('url_link', 'img')
    can_delete = False
    extra = 0

    def img(self, obj):
        return u'<img src="{url}" alt="{url}" />'.format(url=obj.url)

    img.allow_tags = True
    img.short_description = u'Скриншот'

    def url_link(self, obj):
        return u'<a target="_blank" rel="noopener noreferrer" href="{url}">{url}</a>'.format(url=obj.url)

    url_link.allow_tags = True
    url_link.short_description = u'Ссылка на скриншот в MDS'


class PartnerReviewAdminForm(forms.ModelForm):
    redirect_params = forms.CharField(widget=widgets.AdminTextareaWidget())

    class Meta:
        model = PartnerReview
        fields = [
            'partner',
            'result',
            'description',
            'hit_time',
            'review_time',
            'search_depth',
            'point_from',
            'point_to',
            'date_forward',
            'date_backward',
            'klass',
            'adults',
            'children',
            'infants',
            'national_version',
            'price_value',
            'price_currency',
            'price_unixtime',
            'shown_price_value',
            'shown_price_currency',
            'shown_price_unixtime',
            'price_diff_abs',
            'price_diff_rel',
            'price_revise',
            'order_content',
            'revise_data',
            'redirect_params',
            'user_info',
            'query_source',
            'utm_source',
            'utm_campaign',
            'utm_medium',
            'utm_content',
            'wizard_redir_key',
            'wizard_flags',
            'marker',
        ]


class InputFilter(admin.SimpleListFilter):
    template = 'input_filter.html'

    def lookups(self, request, model_admin):
        return ((),)

    def choices(self, changelist):
        all_choice = next(super(InputFilter, self).choices(changelist))
        all_choice['query_parts'] = (
            (k, v)
            for k, v in changelist.get_filters_params().items()
            if k != self.parameter_name
        )
        yield all_choice


class UTMContentFilter(InputFilter):
    parameter_name = 'utm_content'
    title = 'utm_content'

    def queryset(self, request, queryset):
        if self.value() is not None:
            text = self.value()
            return queryset.filter(
                utm_content__contains=text
            )


class CustomPartnerReviewResultFilter(SimpleListFilter):
    title = u'Результат'
    parameter_name = 'result__in'

    def lookups(self, request, model_admin):
        return list(PARTNER_REVIEW_RESULT_CHOICES) + [
            ('problem,error', u'Не прошла')
        ]

    def queryset(self, request, queryset):
        if self.value() is not None:
            return queryset.filter(result__in=self.value().split(','))
        else:
            return queryset


class CustomPartnerReviewPartnersFilter(SimpleListFilter):
    title = u'Партнеры'
    parameter_name = 'partner__in'

    def lookups(self, request, model_admin):
        return [
            (x, x)
            for x in PartnerReview.objects.
            values_list('partner', flat=True).distinct()
        ]

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(partner__in=self.value().split(','))
        else:
            return queryset


def avia_admin_partner_url(code):
    if code.startswith('dohop_'):
        modelurl = 'order/dohopvendor/?dohop_id={dohop_id}'.format(
            dohop_id=int(code.split('_')[1])
        )
    else:
        modelurl = 'order/partner/?code={code}'.format(code=code)

    return 'https://{host}/admin/{modelurl}'.format(
        host=settings.AVIA_ADMIN_HOST,
        modelurl=modelurl
    )


PeriodInfo = namedtuple('PeriodInfo', ['name', 'boundary', 'delta'])


class PartnerReviewAdmin(admin.ModelAdmin):
    form = PartnerReviewAdminForm

    def partner_link(self, obj):
        code = obj.partner
        url = avia_admin_partner_url(code)
        return u'<a href="{url}">{title}</a>'.format(url=url, title=code)

    partner_link.allow_tags = True
    partner_link.short_description = u'Партнёр в админке расписаний'

    def result_flag(self, obj):
        result_icons = {
            'pass': 'icon-yes.svg',
            'problem': 'icon-no.svg',
            'error': 'icon-unknown.svg',
        }

        return u'<img src="/static/admin/img/%s" alt="%s"> %s' % (
            result_icons.get(obj.result, 'icon-alert.svg'),
            obj.result,
            obj.get_result_display()
        )

    def get_search_results(self, request, queryset, search_term):
        return partner_review_filter.search(
            search_term, queryset
        ), False

    result_flag.allow_tags = True
    result_flag.short_description = u'Результат проверки'

    search_fields = ('partners', 'utm_source', 'utm_medium', 'utm_content')

    fieldsets = (
        (u'', {'fields': [
            (
                'result',
                'description',
            ),
            'partner',
        ]}),
        (u'Цена', {'fields': [
            ('shown_price_value', 'shown_price_currency', 'shown_price_unixtime'),
            ('price_value', 'price_currency', 'price_unixtime'),
            'price_revise',
            'price_diff_abs',
            'price_diff_rel',
        ]}),
        (u'Вариант', {'fields': [
            'order_content',
            'view_point_from',
            'view_point_to',
            ('date_forward', 'date_backward'),
            'klass',
            'adults',
            'children',
            'infants',
            'national_version',
            'query_source',
            'utm_source',
            'utm_campaign',
            'utm_medium',
            'utm_content',
            'wizard_redir_key',
            'wizard_flags',
            'marker',
        ]}),
        (u'Проверка', {'fields': [
            'hit_time',
            'review_time',
            'search_depth',
            'revise_data',
            'redirect_params',
            'user_info',
        ]}),
    )

    list_display = (
        'hit_time',
        'partner_link',
        'search_depth',
        'result_flag',
        'description',
        'shown_price_value',
        'price_value',
        'price_diff_abs',
        'price_diff_rel',
        'view_point_from',
        'view_point_to',
        'date_forward',
        'date_backward',
        # 'klass',
        'adults',
        'children',
        'infants',
        # 'national_version',
        'query_source',
        'utm_source',
        'utm_campaign',
        'utm_medium',
        'utm_content',
        'wizard_redir_key',
        'wizard_flags',
        'marker',
    )

    list_filter = (
        CustomPartnerReviewPartnersFilter,
        CustomPartnerReviewResultFilter,
        'adults',
        ('hit_time', AviaDateRangeFilter),
        'children',
        'infants',
        'national_version',
        'klass',
        'query_source',
        'utm_source',
        'utm_medium',
        UTMContentFilter,
        'wizard_flags',
    )

    readonly_fields = ('view_point_from', 'view_point_to', 'shown_price_unixtime', 'price_unixtime')

    inlines = (ScreenshotsInline,)

    def _view_point_title(self, point_key):
        point = get_point_by_point_key(point_key)
        return point.title if point else '-'

    def view_point_from(self, obj):
        return self._view_point_title(obj.point_from)
    view_point_from.short_description = u'Откуда'

    def view_point_to(self, obj):
        return self._view_point_title(obj.point_to)
    view_point_to.short_description = u'Куда'

    def get_urls(self):
        urls = super(PartnerReviewAdmin, self).get_urls()
        my_urls = [
            url(r'^stat/$', self.admin_site.admin_view(self.stat_view)),
        ]
        return my_urls + urls

    def stat_view(self, request):
        model = self.model
        opts = model._meta
        media = self.media

        now = datetime.now().replace(second=0, microsecond=0, tzinfo=None)

        def add_month(dt, months):
            return dt + relativedelta(months=months)

        periods = OrderedDict((
            ('h', PeriodInfo('Час', now - timedelta(hours=1), timedelta(minutes=15))),
            ('3h', PeriodInfo('3 часа', now - timedelta(hours=3), timedelta(minutes=15))),
            ('12h', PeriodInfo('12 часов', now - timedelta(hours=12), timedelta(minutes=30))),
            ('24h', PeriodInfo('24 часа', now - timedelta(hours=24), timedelta(hours=1))),
            ('3d', PeriodInfo('3 дня', now - timedelta(days=3), timedelta(hours=2))),
            ('w', PeriodInfo('Неделя', now - timedelta(weeks=1), timedelta(hours=6))),
            ('m', PeriodInfo('Месяц', add_month(now, -1), timedelta(hours=6))),
            ('3m', PeriodInfo('3 месяца', add_month(now, -3), timedelta(days=3))),
            ('6m', PeriodInfo('6 месяцев', add_month(now, -6), timedelta(weeks=1))),
            ('y', PeriodInfo('Год', now.replace(year=now.year-1), timedelta(days=30))),
        ))

        period = request.GET.get('period')

        if period not in periods:
            period = '12h'

        periods_links = [
            ('?period=%s' % code if code != period else None, period_info.name)
            for code, period_info in periods.iteritems()
        ]
        period_info = periods[period]
        period_boundary = period_info.boundary

        reviews = model.objects.filter(review_time__gt=period_boundary)

        partners = reviews.values_list('partner', flat=True).distinct()

        reviews_counts = dict(
            reviews.
            values_list('partner').annotate(Count('partner')).order_by()
        )

        warnings_counts = dict(
            reviews.filter(result='pass').exclude(price_diff_abs=0).
            values_list('partner').annotate(Count('partner')).order_by()
        )

        problems_counts = dict(
            reviews.filter(result='problem').
            values_list('partner').annotate(Count('partner')).order_by()
        )

        errors_counts = dict(
            reviews.filter(result='error').
            values_list('partner').annotate(Count('partner')).order_by()
        )

        def partnerreview_link(p, **kwargs):
            kwargs['partner__in'] = p
            return '/admin/data/partnerreview/?' + urlencode(kwargs)

        def json_stat(partner):
            p = partner
            reviews = reviews_counts.get(p, 0)

            return {
                'partner': {
                    'code': p,
                    'raspadmin_link': avia_admin_partner_url(p),
                },
                'reviews_count': reviews,
                'warnings_count': int(100 * warnings_counts.get(p, 0) / reviews),
                'problems_count': int(100 * problems_counts.get(p, 0) / reviews),
                'errors_count': int(100 * errors_counts.get(p, 0) / reviews),
                'reviews_count_link': partnerreview_link(
                    p, review_time__gt=period_boundary,
                ),
                'warnings_count_link': partnerreview_link(
                    p, review_time__gt=period_boundary,
                    result='pass', description='Цена почти совпадает'
                ),
                'problems_count_link': partnerreview_link(
                    p, review_time__gt=period_boundary,
                    result='problem'
                ),
                'errors_count_link': partnerreview_link(
                    p, review_time__gt=period_boundary,
                    result='error'
                ),
            }

        def _get_graph_stat():
            def result_count_by_partners(reviews, result):
                return dict(reviews.filter(result=result).values_list('partner').annotate(Count('partner')).order_by())

            def split_by_periods():
                interval = [period_boundary, period_boundary + period_info.delta]
                while interval[0] < now:
                    _reviews = reviews.filter(review_time__range=interval)
                    partners = _reviews.values_list('partner', flat=True).distinct()
                    pass_counts = result_count_by_partners(_reviews, 'pass')
                    problems_counts = result_count_by_partners(_reviews, 'problem')

                    interval_end_ts = calendar.timegm(interval[1].timetuple()) * 1000

                    for p in partners:
                        pass_count = pass_counts.get(p, 0)
                        problems_count = problems_counts.get(p, 0)
                        if pass_count or problems_count:
                            yield [str(p), pass_count, problems_count, interval_end_ts]

                    interval = map(lambda x: x + period_info.delta, interval)

            detailed_stat = list(split_by_periods())

            by_partners_info = defaultdict(list)
            summary_by_intervals_info = {}
            for partner, pass_count, error_count, interval_end in detailed_stat:
                info = [pass_count, error_count, interval_end]
                by_partners_info[partner].append(info)

                if interval_end not in summary_by_intervals_info:
                    summary_by_intervals_info[interval_end] = copy(info)
                else:
                    summary_by_intervals_info[interval_end][0] += pass_count
                    summary_by_intervals_info[interval_end][1] += error_count

            summary_info = sorted(summary_by_intervals_info.values(), key=lambda x: x[2])
            by_partners_info = sorted(by_partners_info.iteritems())

            def info_to_graph_data(info):
                return [[i, float(neg) / (pos + neg)] for pos, neg, i in info]

            all_partners_stat = [{
                'idx': 0,
                'label': 'all_partners',
                'tooltip_info': summary_info,
                'data': info_to_graph_data(summary_info)
            }]
            by_partners_stat = [{
                'idx': idx,
                'label': partner,
                'tooltip_info': info_,
                'data': info_to_graph_data(info_),
            } for idx, (partner, info_) in enumerate(by_partners_info, start=1)]

            return {'detailed_by_partners_stat': all_partners_stat + by_partners_stat}

        context = dict(
            # Include common variables for rendering the admin template.
            self.admin_site.each_context(request),
        )
        context.update({
            'title': u'Статистика проверок цен от партнёров',
            'media': media,
            'app_label': opts.app_label,
            'opts': opts,
            'has_absolute_url': hasattr(self.model, 'get_absolute_url'),
            'content_type_id': ContentType.objects.get_for_model(self.model).id,
            'stat': map(json_stat, partners),
            'periods_links': periods_links,
        })

        context.update(_get_graph_stat())

        return TemplateResponse(
            request,
            'admin/%s/stat.html' % opts.object_name.lower(),
            context,
            current_app=self.admin_site.name
        )


admin.site.register(PartnerReview, PartnerReviewAdmin)
