# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
import re

from datetime import date, time, datetime, timedelta
from functools import wraps

from django.http import HttpResponse

VALID_CALLBACK_RE = re.compile(r'^[A-Za-z\d_]*$')


def jsonp_response(func):
    @wraps(func)
    def wrapper(request, *args, **kwargs):
        data = func(request, *args, **kwargs)

        if isinstance(data, HttpResponse):
            return data

        return jsonp_wrap(request, data, serialize=serialize)

    return wrapper


def csv_response(func):
    @wraps(func)
    def wrapper(request, *args, **kwargs):
        data = func(request, *args, **kwargs)

        if isinstance(data, HttpResponse):
            return data

        response = HttpResponse(data, content_type="text/plain; charset=utf-8")
        # show in browser. disable download
        # response['Content-Disposition'] = 'attachment; filename="redirects.txt"'

        return response

    return wrapper


def serialize(obj):
    if isinstance(obj, datetime):
        return json_datetime_aware(obj)

    return rich_serialize(obj)


def rich_serialize(obj):
    if hasattr(obj, '__json__'):
        return obj.__json__()

    if isinstance(obj, datetime):
        return obj.strftime("%Y-%m-%d %H:%M:%S")

    if isinstance(obj, date):
        return obj.strftime("%Y-%m-%d")

    if isinstance(obj, timedelta):
        return obj.seconds + obj.days * 84600

    if isinstance(obj, time):
        return obj.strftime('%H:%M')

    raise TypeError(repr(obj) + " is not JSON serializable")


def json_datetime_aware(dt):
    if not dt:
        return None

    return {
        'local': dt.strftime("%a, %d %b %Y %H:%M:%S GMT"),
        'offset': timedelta2minutes(dt.utcoffset()),
    }


def timedelta2minutes(delta):
    return delta.total_seconds() / 60


def jsonp_wrap(request, data, serialize=None):
    serialized = json.dumps(
        data,
        default=serialize or rich_serialize,
        ensure_ascii=False,
        separators=(',', ':')
    )

    return HttpResponse(serialized, content_type="application/json; charset=utf-8")


def is_valid_jsonp_callback(callback):
    return VALID_CALLBACK_RE.match(callback) and len(callback) <= 60
