# -*- coding: utf-8 -*-
from django.db import models
from django.conf import settings


SERVICES = {
    '42': u'Грелка',
    'admin': u'Админка Авиабилетов',
    'alice': u'Алиса',
    'api_avia': u'Публичное апи',
    'avia-travel': u'Десктоп авиа в портале',
    'boy': u'Book on Yandex',
    'email_subscriptions': u'Подписки на изменение цен',
    'mavia-travel': u'Тач авиа в портале',
    'm_avia': u'Тач версия',
    'rasp_api_public': u'Публичное апи расписаний',
    'rasp_mobile': u'PDA версия расписаний',
    'rasp_morda': u'Старая десктопная версия расписаний',
    'rasp_morda_backend': u'Бэкэнд расписания',
    'rasp_touch': u'Тач версия расписаний',
    'sovetnik': u'Советник',
    'ticket': u'Десктопная версия',
    'tours': u'Яндекс.Путешествия',
    'welcome2018': u'Чемпионат мира по футболу 2018',
    'wizard': u'Колдунщик авиабилетов',
    'yeah': u'Грелка 2.0',
}


class Error500(models.Model):
    eventdate = models.DateField(
        db_index=True
    )

    code = models.PositiveIntegerField(
        default=0,
    )

    min_eventtime = models.DateTimeField(blank=True, null=True)
    max_eventtime = models.DateTimeField(blank=True, null=True)

    count = models.PositiveIntegerField(
        default=0
    )

    request = models.CharField(
        max_length=255,
    )

    class Meta:
        app_label = 'data'
        ordering = ['-eventdate', '-count', 'code']


class UsersSearch(models.Model):
    service = models.CharField(
        max_length=255,
        choices=[(p, p) for p in SERVICES.keys()],
        blank=False, null=False, verbose_name=u"Сервис",
        db_index=False
    )

    eventdate = models.DateTimeField(
        blank=True, null=True, verbose_name=u"Время",
        db_index=False
    )

    count = models.IntegerField(
        blank=False, null=False, default=0,
        verbose_name=u"Кол-во"
    )

    national_version = models.CharField(
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN,
        verbose_name=u"Нац. версия",
        null=True,
        default='ru'
    )

    pp = models.CharField(
        max_length=32,
        verbose_name=u"pp",
        null=False,
        default=''
    )

    class Meta:
        unique_together = ("service", "eventdate", "national_version", "pp")
        app_label = 'data'
        verbose_name = u"Пользовательский поиск"
        verbose_name_plural = u"Пользовательские поиски"


class UserRedirect(models.Model):
    service = models.CharField(
        max_length=255,
        choices=[(p, p) for p in SERVICES.keys()],
        blank=False, null=False, verbose_name=u"Сервис",
        db_index=False
    )

    eventdate = models.DateTimeField(
        blank=True, null=True, verbose_name=u"Время",
        db_index=False
    )

    count = models.IntegerField(
        blank=False, null=False, default=0,
        verbose_name=u"Кол-во",
        db_index=False
    )

    partner_code = models.CharField(
        max_length=32,
        verbose_name=u"Код партнера",
        db_index=False
    )

    national_version = models.CharField(
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN,
        verbose_name=u"Нац. версия",
        null=True,
        default='ru'
    )

    pp = models.CharField(
        max_length=32,
        verbose_name=u"pp",
        null=False,
        default=''
    )

    class Meta:
        unique_together = ("national_version", "service", "eventdate", "partner_code", "pp")
        app_label = 'data'
        verbose_name = u"Пользовательский переход"
        verbose_name_plural = u"Пользовательские переходы"


class YandexUID(models.Model):
    service = models.CharField(
        max_length=255,
        choices=[(p, p) for p in SERVICES.keys()],
        blank=False, null=False, verbose_name=u"Сервис",
        db_index=False
    )

    eventdate = models.DateField(
        blank=True, null=True, verbose_name=u"Время",
        db_index=False
    )

    count = models.IntegerField(
        blank=False, null=False, default=0,
        verbose_name=u"Кол-во"
    )

    national_version = models.CharField(
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN,
        verbose_name=u"Нац. версия",
        null=True,
        default='ru'
    )

    class Meta:
        unique_together = ("service", "eventdate", "national_version")
        app_label = 'data'
        verbose_name = u"yandexUID"
        verbose_name_plural = u"yandexUIDs"


class ServiceMetric(models.Model):
    service = models.CharField(
        max_length=255,
        choices=[(p, p) for p in SERVICES.keys()],
        blank=False, null=False, verbose_name=u"Сервис",
    )

    event_datetime = models.DateTimeField(
        blank=False, null=False, verbose_name=u"Время",
    )

    event_day = models.PositiveIntegerField(
        blank=False, null=False, verbose_name=u"День",
    )

    event_month = models.PositiveIntegerField(
        blank=False, null=False, verbose_name=u"Месяц",
    )

    event_year = models.PositiveIntegerField(
        blank=False, null=False, verbose_name=u"Год",
    )

    event_name = models.CharField(
        max_length=128,
        blank=False, null=False, verbose_name=u"Метрика",
    )

    value = models.IntegerField(
        blank=False, null=False, verbose_name=u"Значение",
    )

    class Meta:
        unique_together = ("service", "event_name", "event_datetime")
        index_together = [
            ["service", "event_name", 'event_year', 'event_month', 'event_day'],
            ["event_name", "event_datetime"],
        ]
        app_label = 'data'
        verbose_name = u"метрика"
        verbose_name_plural = u"метрики"


class AbstractPartnerQuery(models.Model):
    service = models.CharField(
        max_length=255,
        choices=[(p, p) for p in SERVICES.keys()],
        blank=False, null=False, verbose_name=u"Сервис",
    )

    query_datetime = models.DateTimeField(
        blank=False, null=False, verbose_name=u"Время",
        db_index=True,
    )

    query_status = models.CharField(
        max_length=128,
        blank=False, null=False, verbose_name=u"Статус ответа",
    )

    partner_code = models.CharField(
        max_length=128,
        blank=False, null=False, verbose_name=u"Импортер",
    )

    count = models.PositiveIntegerField(
        blank=False, null=False, verbose_name=u"Кол-во",
    )

    avg_query_time = models.PositiveIntegerField(
        blank=False, null=False, verbose_name=u"Ср. время ответа",
        default=0
    )

    class Meta:
        abstract = True


class PartnerQuery(AbstractPartnerQuery):
    class Meta:
        index_together = [
            ["service", "query_datetime", 'partner_code'],
            ["query_status"],
        ]
        app_label = 'data'
        verbose_name = u"опрос парнера"
        verbose_name_plural = u"опросы парнеров"


class ArchivePartnerQuery(AbstractPartnerQuery):
    id = models.IntegerField(primary_key=True)

    class Meta:
        app_label='data'
        db_table = 'data_partnerquery_archive'


class RedirectHeatMap(models.Model):
    eventdate = models.DateTimeField(
        blank=True, null=True, verbose_name=u"Время",
        db_index=False
    )

    national_version = models.CharField(
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN,
        verbose_name=u"Нац. версия",
        null=True,
        default='ru'
    )

    count = models.IntegerField(
        blank=False, null=False, default=0,
        verbose_name=u"Кол-во",
        db_index=False
    )

    geo_id = models.IntegerField(
        blank=False, null=False, default=0,
        verbose_name=u"geo_id",
        db_index=False
    )

    lon = models.FloatField(
        verbose_name=u"Долгота",
    )

    lat = models.FloatField(
        verbose_name=u"Широта",
    )

    country = models.CharField(
        verbose_name=u"Страна",
        max_length=128,
        null=True,
        default='ru'
    )

    class Meta:
        unique_together = ("national_version", "eventdate", "geo_id")
        app_label = 'data'
        verbose_name = u"Пользовательский переход"
        verbose_name_plural = u"Пользовательские переходы"


class UTM(models.Model):
    source = models.CharField(max_length=255)
    campaign = models.CharField(max_length=255)
    medium = models.CharField(max_length=255)

    class Meta:
        unique_together = ("source", "campaign", "medium")
        app_label = 'data'
        verbose_name = u"UTM метка"
        verbose_name_plural = u"UTM метки"


class UTM_ReportsHistory(models.Model):
    report_date = models.DateField(
        blank=False, null=False, verbose_name=u"Дата события"
    )

    yaml_file = models.CharField(
        blank=False, null=False, max_length=128, verbose_name=u"Yaml файл"
    )

    eventdatetime = models.DateTimeField(
        blank=True, null=True, verbose_name=u"Дата события"
    )

    success = models.BooleanField(
        blank=False, null=False, verbose_name=u"Успешно"
    )

    last_error = models.TextField(
        blank=True, null=True, verbose_name=u"Текст ошшибки"
    )

    report_content = models.TextField(
        blank=True, null=True, verbose_name=u"Содержимое отчета"
    )

    class Meta:
        unique_together = ("report_date", "yaml_file")
        ordering = ['-report_date', '-eventdatetime']
        app_label = 'data'
        verbose_name = u"История adwords"
        verbose_name_plural = u"История adwords"


class UTM_adwords(models.Model):
    eventdate = models.DateField(
        blank=True, null=True, verbose_name=u"Дата",
        db_index=True
    )

    utm = models.ForeignKey(
        UTM,
        verbose_name=u'UTM',
        related_name='+'
    )

    cost = models.FloatField(
        blank=False, null=False, default=0, verbose_name=u"Цена кликов"
    )

    clicks = models.IntegerField(
        blank=False, null=False, default=0, verbose_name=u"Цена кликов"
    )

    class Meta:
        app_label = 'data'
        verbose_name = u"данные adwords"
        verbose_name_plural = u"данные adwords"


class UTM_incoming(models.Model):
    eventdate = models.DateField(
        blank=True, null=True, verbose_name=u"Дата",
        db_index=False
    )

    utm = models.ForeignKey(
        UTM,
        verbose_name=u'UTM',
        related_name='+'
    )

    yandexuid = models.CharField(max_length=32)

    price = models.FloatField(
        blank=False, null=False, default=0, verbose_name=u"Цена клика"
    )

    class Meta:
        index_together = [["eventdate", "yandexuid"], ["yandexuid"]]
        app_label = 'data'
        verbose_name = u"входящая utm"
        verbose_name_plural = u"входящие utm"


class UTM_redirect(models.Model):
    eventdate = models.DateField(
        blank=True, null=True, verbose_name=u"Дата",
        db_index=False
    )

    utm = models.ForeignKey(
        UTM,
        verbose_name=u'UTM',
        related_name='+'
    )

    cohort = models.PositiveSmallIntegerField(verbose_name=u'UTM')

    price = models.FloatField(
        blank=False, null=False, default=0, verbose_name=u"Цена перехода"
    )

    yandexuid = models.CharField(max_length=32)

    class Meta:
        index_together = [["eventdate", "yandexuid"], ["yandexuid"]]
        app_label = 'data'
        verbose_name = u"исходящая utm"
        verbose_name_plural = u"исходящие utm"


class FlexibleCache(models.Model):
    eventdate = models.DateField(
        verbose_name=u"Дата",
        blank=False, null=False, db_index=False
    )

    percent = models.PositiveSmallIntegerField(
        verbose_name=u"Процент попаданий",
        blank=False, null=False, db_index=False
    )

    count = models.PositiveIntegerField(
        verbose_name=u"Кол-во",
        blank=False, null=False, db_index=False
    )

    national_version = models.CharField(
        u'Нац. версия',
        choices=settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN,
        db_index=False
    )

    class Meta:
        unique_together = ('national_version', 'eventdate', 'percent')
        ordering = ['national_version', 'eventdate', 'percent', 'count']


class DynamicCache(models.Model):
    eventdate = models.DateField(
        verbose_name=u"Дата",
        blank=False, null=False
    )

    percent = models.PositiveSmallIntegerField(
        verbose_name=u"Процент попаданий",
        blank=False, null=False
    )

    count = models.PositiveIntegerField(
        verbose_name=u"Кол-во",
        blank=False, null=False
    )

    month_year = models.CharField(
        verbose_name=u"Месяц и год",
        max_length=7,
        blank=False, null=False
    )

    national_version = models.CharField(
        u'Нац. версия',
        choices=settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN,
    )

    class Meta:
        unique_together = ('national_version', 'month_year', 'eventdate')
        ordering = ['national_version', 'month_year', 'eventdate', ]
        app_label = 'data'


class RivalPosition(models.Model):
    eventdate = models.DateField(
        verbose_name=u"Дата",
    )
    rival_name = models.CharField(
        verbose_name=u"Название сайта",
        max_length=64, null=False, blank=False
    )

    partner = models.CharField(
        verbose_name=u"Партнер",
        max_length=64, null=False, blank=False
    )

    direction = models.CharField(
        verbose_name=u"Направление",
        max_length=128, null=False, blank=False
    )
    passengers = models.CharField(
        verbose_name=u"Состав пассажиров",
        max_length=5, null=False, blank=False
    )
    one_way = models.BooleanField(
        verbose_name=u"Билет в одну сторону",
    )
    position = models.PositiveIntegerField(
        verbose_name=u"Позиция",
        null=False, blank=False
    )
    minimal_price = models.PositiveIntegerField(
        verbose_name=u"Мин. цена",
        null=False, blank=False, default=0
    )
    avia_company = models.CharField(
        verbose_name=u"Авиа-компания",
        max_length=255, null=False, blank=False, default="",
    )

    cheaper = models.BooleanField(
        null=False, blank=False, default=False,
    )

    class Meta:
        index_together = [
            ['eventdate', 'rival_name'],
            ['rival_name']
        ]
        ordering = ['eventdate', 'rival_name', 'direction', 'passengers', 'one_way', ]
        app_label = 'data'


class OhmUtm(models.Model):
    source = models.CharField(max_length=255)
    campaign = models.CharField(max_length=255)
    medium = models.CharField(max_length=255)
    term = models.CharField(max_length=255)

    def tuple_key(self):
        return (
            self.source.encode('utf-8'),
            self.campaign.encode('utf-8'),
            self.medium.encode('utf-8'),
            self.term.encode('utf-8'),
        )

    class Meta:
        unique_together = ("source", "campaign", "medium", "term")
        app_label = 'data'
        verbose_name = u"OHM UTM метка"
        verbose_name_plural = u"OHM UTM метки"


class OhmIncoming(models.Model):
    eventdate = models.DateField(
        blank=False, null=False, verbose_name=u"Дата"
    )

    eventdatetime = models.DateTimeField(
        blank=False, null=False, verbose_name=u"Дата и время",
        db_index=True,
    )

    utm = models.ForeignKey(
        OhmUtm,
        verbose_name=u'UTM',
        related_name='+'
    )

    yandexuid = models.CharField(max_length=32)

    version = models.PositiveSmallIntegerField(default=1, verbose_name=u'Версия данных')

    class Meta:
        index_together = [
            ["version", "eventdate", "yandexuid"],
            ["version", "yandexuid", "eventdatetime"],
            ["version", "eventdatetime"],
            ["eventdatetime"]
        ]
        app_label = 'data'
        verbose_name = u"OHM входяшая UTM"
        verbose_name_plural = u"OHM входящие UTM"


class OhmRedirect(models.Model):
    eventdate = models.DateField(
        blank=False, null=False, verbose_name=u"Дата"
    )

    eventdatetime = models.DateTimeField(
        blank=False, null=False, verbose_name=u"Дата и время"
    )

    yandexuid = models.CharField(max_length=32)

    version = models.PositiveSmallIntegerField(default=1, verbose_name=u'Версия данных')

    incoming = models.ForeignKey(
        OhmIncoming,
        null=True, verbose_name=u'Incoming',
    )

    click_price = models.DecimalField(null=False, default=19.4, decimal_places=2, max_digits=6)

    class Meta:
        index_together = [
            ["version", "eventdate", "yandexuid"],
            ["version", "yandexuid", "eventdatetime"],
            ["version", "eventdatetime"],
            ["version", "id"],
            ["version", "incoming"],
            ["eventdatetime"]
        ]
        app_label = 'data'
        verbose_name = u"OHM редирект"
        verbose_name_plural = u"OHM редиректы"


class OhmAdwords(models.Model):
    eventdate = models.DateField(
        blank=False, null=False, verbose_name=u"Дата",
        db_index=True
    )

    utm = models.ForeignKey(
        OhmUtm,
        verbose_name=u'UTM',
        related_name='+'
    )

    cost = models.FloatField(
        blank=False, null=False, default=0, verbose_name=u"Цена кликов"
    )

    clicks = models.IntegerField(
        blank=False, null=False, default=0, verbose_name=u"Кол-во кликов"
    )

    class Meta:
        app_label = 'data'
        index_together = [['eventdate', 'utm']]
        verbose_name = u"Ohm данные adwords"
        verbose_name_plural = u"Ohm данные adwords"


class UserGoalStat(models.Model):
    eventdate = models.DateField(
        blank=False, null=False, verbose_name=u"Дата",
        db_index=True
    )

    yandexuid = models.BigIntegerField(
        blank=False, null=False, verbose_name=u"YandexUID",
    )

    user_agent = models.CharField(
        blank=False, null=False, verbose_name=u"Браузер",
        max_length=128
    )

    goals_count = models.PositiveIntegerField(
        blank=False, null=False, verbose_name=u"Достижения",
    )

    avia_count = models.PositiveIntegerField(
        blank=False, null=False, verbose_name=u"Редиректы",
    )

    class Meta:
        app_label = 'data'
        index_together = [['eventdate', 'yandexuid']]
        verbose_name = u"достижение целей"
        verbose_name_plural = u"достижения целей"


class Setting(models.Model):
    key = models.CharField(max_length=128)
    value = models.CharField(max_length=128)

    class Meta:
        app_label = 'data'
        unique_together = ['key']
        verbose_name = u"Настройка"
        verbose_name_plural = u"Настройки"
