# -*- coding: utf-8 -*-
from __future__ import division
import base64
import json
import logging
from datetime import datetime
import time

from django.db import models
from django.conf import settings

from travel.avia.stat_admin.lib import s3_sync


yt_logger = logging.getLogger('yt.partner_review')
log = logging.getLogger(__name__)


class TrimmedCharField(models.CharField):
    def pre_save(self, instance, add):
        value = models.CharField.pre_save(self, instance, add)
        if value is not None:
            if isinstance(value, basestring):
                value = value.strip()
        setattr(instance, self.attname, value)
        return value


PARTNER_REVIEW_RESULT_CHOICES = (
    ('', u'Не известно'),
    ('pass', u'Успешно'),
    ('problem', u'Проблема'),
    ('problem_1_10', u'Незначительная проблема'),
    ('error', u'Ошибка'),
    ('can_not_fetch_review_error', u'Ошибка проверки'),
    ('can_not_fetch_price_error', u'Ошибка получения цены'),
    ('ticket_changed', u'Цена/наличие поменялись'),
)

PARTNER_REVIEW_SHOWN_RESULT_CHOICES = (
    ('', u'Не известно'),
    ('pass', u'Успешно'),
    ('problem', u'Проблема'),
    ('problem_1_10', u'Незначительная проблема'),
    ('error', u'Ошибка'),
    ('shown_price_is_absent', u'Показанная цена не задана'),
    ('can_not_fetch_review_error', u'Ошибка проверки'),
    ('can_not_fetch_price_error', u'Ошибка получения цены'),
    ('ticket_changed', u'Цена/наличие поменялись'),
)

CURRENCY_CHOICES = (
    ('RUR', u'рубли'),
    ('USD', u'доллары'),
    ('EUR', u'евро'),
    ('UAH', u'гривни'),
    ('KZT', u'тенге'),
    ('BYR', u'белорусские рубли'),
    ('TRY', u'турецкие лиры'),
)


SECONDS_IN_DAY = 60*60*24


class PartnerReview(models.Model):
    partner = TrimmedCharField(
        u'Партнер', max_length=30, null=False, blank=False
    )

    result = models.CharField(
        u'Результат проверки',
        choices=PARTNER_REVIEW_RESULT_CHOICES,
        max_length=30, null=False, blank=True, default=''
    )

    shown_result = models.CharField(
        u'Результат сравнения показанной цены и цены при ревайзе',
        choices=PARTNER_REVIEW_SHOWN_RESULT_CHOICES,
        max_length=30, null=False, blank=True, default=''
    )

    description = TrimmedCharField(
        u'Описание', max_length=255, null=False, blank=True, default=''
    )

    hit_time = models.DateTimeField(
        u'Время перехода пользователя', blank=False, null=False, db_index=True
    )

    review_time = models.DateTimeField(
        u'Время проверки', blank=True, null=True
    )

    point_from = TrimmedCharField(
        u'Откуда', max_length=12, null=False, blank=False
    )

    point_to = TrimmedCharField(
        u'Куда', max_length=12, null=False, blank=False
    )

    date_forward = models.DateField(u'Дата туда', null=False, blank=False)
    date_backward = models.DateField(u'Дата обратно', null=True, blank=True)

    klass = TrimmedCharField(
        u'Класс обслуживания',
        max_length=12, null=False, blank=False, default=''
    )

    adults = models.IntegerField(u'Взрослых', null=False, blank=False, default=1)
    children = models.IntegerField(u'Детей', null=False, blank=False, default=0)
    infants = models.IntegerField(u'Младенцев', null=False, blank=False, default=0)

    national_version = TrimmedCharField(
        u'Национальная версия', null=False, blank=False, default='',
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN
    )

    price_value = models.FloatField(u'Цена на выдаче', blank=False, null=False)
    price_currency = models.CharField(
        u'Валюта', choices=CURRENCY_CHOICES,
        max_length=3, null=False, blank=False
    )
    price_unixtime = models.DateTimeField(
        u'Возраст кэша из базы', null=True, blank=True
    )

    shown_price_value = models.FloatField(u'Цена на кнопке', blank=True, null=True)
    shown_price_currency = models.CharField(
        u'Валюта цены на кнопке', choices=CURRENCY_CHOICES,
        max_length=3, null=True, blank=True
    )
    shown_price_unixtime = models.DateTimeField(
        u'Возраст кэша с кнопки', null=True, blank=True
    )

    shown_price_diff_abs = models.FloatField(
        u'Абсолютная разница в цене между показанной ценой и ценой, полученной при ревайзе',
        blank=True, null=True
    )

    shown_price_diff_rel = models.FloatField(
        'Относительная разница в цене между показанной ценой и ценой, полученной при ревайзе',
        blank=True, null=True
    )

    price_revise = models.FloatField(u'Цена на сайте партнёра', blank=True, null=True)
    currency_revise = models.CharField(
        u'Валюта на сайте партнёра',
        max_length=12, null=True, blank=True
    )

    price_diff_abs = models.FloatField(
        u'Абсолютная разница в цене', blank=True, null=True
    )

    price_diff_rel = models.FloatField(
        u'Относительная разница в цене, %', blank=True, null=True
    )

    order_content = models.TextField(u'Параметры билета')
    revise_data = models.TextField(
        u'Ответ revise', default='', blank=True, null=True
    )

    redirect_params = models.TextField(u'Параметры перехода')
    user_info = models.TextField(
        u'Данные пользователя', default='', blank=True, null=True
    )

    search_depth = models.IntegerField(
        u'Глубина поиска, дней',
        default=None, blank=True, null=True, db_index=True
    )

    utm_source = TrimmedCharField(
        u'utm_source. Откуда переход',
        null=True, blank=True, default='', max_length=48, db_index=True
    )
    utm_campaign = TrimmedCharField(
        u'utm_campaign.',
        null=True, blank=True, default='', max_length=250, db_index=True
    )
    utm_medium = TrimmedCharField(
        u'utm_medium.',
        null=True, blank=True, default='',
        max_length=250, db_index=True
    )
    utm_content = TrimmedCharField(
        u'utm_content.',
        null=True, blank=True, default='',
        max_length=250, db_index=True
    )
    wizard_redir_key = TrimmedCharField(
        u'wizard_redir_key.',
        null=True, blank=True, default='',
        max_length=250, db_index=True
    )
    wizard_flags = TrimmedCharField(
        u'wizard_flags.',
        null=True, blank=True, default='',
        max_length=250, db_index=True
    )
    query_source = TrimmedCharField(
        u'query_source. Источник инициализации опроса',
        null=True, blank=True, default='', max_length=48, db_index=True
    )
    marker = TrimmedCharField(
        u'marker',
        null=True, blank=True, default='', max_length=250, db_index=True
    )

    def save(self, *args, **kwargs):
        search_depth = (self.date_forward - self.hit_time.date())
        self.search_depth = search_depth.total_seconds() // SECONDS_IN_DAY
        super(PartnerReview, self).save(*args, **kwargs)

        try:
            self.save_to_yt()
        except Exception as e:
            log.exception('Can not save to yt log', e)

    def _safe_parse_price(self, price, price_type):
        if price is None:
            return None
        try:
            return float(price)
        except Exception as e:
            log.error(
                'Can not parse price [%r] with type [%r]. Error: %r',
                price, price_type, e
            )
            return None

    def save_to_yt(self):
        unix_time = datetime.utcnow() - datetime(1970, 1, 1)
        date_backward = None
        if self.date_backward:
            date_backward = self.date_backward.strftime('%Y-%m-%d')

        price_unixtime = None
        if self.price_unixtime:
            price_unixtime = int(time.mktime(self.price_unixtime.timetuple()))

        shown_price_unixtime = None
        if self.shown_price_unixtime:
            shown_price_unixtime = int(time.mktime(self.shown_price_unixtime.timetuple()))

        yt_logger.info('%s', json.dumps({
            'unixtime': int(unix_time.total_seconds()),
            'partner': self.partner,
            'result': self.result,
            'hit_time': self.hit_time.strftime('%Y-%m-%d %H:%M:%S'),
            'review_time': self.review_time.strftime('%Y-%m-%d %H:%M:%S'),
            'point_from': self.point_from,
            'point_to': self.point_to,
            'date_forward': self.date_forward.strftime('%Y-%m-%d'),
            'date_backward': date_backward,
            'klass': self.klass,
            'adults': self.adults,
            'children': self.children,
            'infants': self.infants,
            'national_version': self.national_version,
            'price': self._safe_parse_price(self.price_value, 'price'),
            'currency': self.price_currency,
            'price_unixtime': price_unixtime,
            'price_revise': self.price_revise,
            'currency_revise': self.currency_revise,
            'price_diff_abs':  self._safe_parse_price(
                self.price_diff_abs,
                'price_diff_abs'
            ),
            'price_diff_rel': self._safe_parse_price(
                self.price_diff_rel,
                'price_diff_rel'
            ),
            'order_content': self.order_content,
            'revise_data': self.revise_data,
            'search_depth': self.search_depth,
            'utm_source': self.utm_source,
            'query_source': self.query_source,
            'utm_medium': self.utm_medium,
            'utm_campaing': self.utm_campaign,
            'utm_content': self.utm_content,
            'wizard_redir_key': self.wizard_redir_key,
            'wizard_flags': self.wizard_flags,

            'shown_price_diff_abs': self.shown_price_diff_abs,
            'shown_price_diff_rel': self.shown_price_diff_rel,
            'shown_result': self.shown_result,
            'shown_price': self.shown_price_value,
            'shown_currency': self.shown_price_currency,
            'shown_price_unixtime': shown_price_unixtime,
            'marker': self.marker,
        }))

    def __unicode__(self):
        return u'[{} {} {}]'.format(
            self.partner,
            self.result or '-',
            self.hit_time.strftime('%Y.%m.%d %H:%M:%S')
        )

    class Meta:
        verbose_name = u'Проверка страницы партнёра'
        verbose_name_plural = u'Проверки страниц партнёров'
        app_label = 'data'
        index_together = [
            ['partner'],
            ['result'],
            ['review_time', 'result']
        ]


class ScreenShot(models.Model):
    url = models.CharField(
        u'Ссылка на скриншот в MDS',
        max_length=128,
        null=False,
        blank=False,
    )

    idx = models.IntegerField(u'Номер скриншота', null=False, blank=False, default=0)

    partner_review = models.ForeignKey(
        PartnerReview,
        verbose_name=u'Проверка',
        blank=False,
        null=False,
        related_name='+',
    )

    @classmethod
    def create(cls, screen_shot, partner_review_id, id_):
        partner_review = PartnerReview.objects.get(id=partner_review_id)
        s3_key = ScreenShot._format_key(partner_review, id_)
        url = cls._mds_upload(screen_shot, s3_key)

        return cls(url=url, partner_review=partner_review, idx=id_)

    @staticmethod
    def _mds_upload(screen_shot, s3_key):
        try:
            png_screen_shot = base64.b64decode(screen_shot)
            content_type = 'image/png'
            bucket = s3_sync.get_bucket()
            return s3_sync.upload_binary(bucket, png_screen_shot, s3_key, content_type)
        except s3_sync.S3SyncError as e:
            log.warning('Can not upload screenshot. key: %s Error: %r', s3_key, e)
            return None

    @staticmethod
    def _format_key(partner_review, id_):
        return '/'.join(
            map(str, (partner_review.partner, partner_review.id, id_))
        )

    def save(self, *args, **kwargs):
        super(ScreenShot, self).save(*args, **kwargs)
