# -*- coding: utf-8 -*-
import calendar
import csv
import logging
import re
import requests

from collections import defaultdict
from cStringIO import StringIO
from datetime import date, datetime, timedelta
from dateutil.relativedelta import relativedelta

from travel.avia.stat_admin.data.httpresponses import csv_response, jsonp_response
from travel.avia.stat_admin.data.models import (
    UserRedirect, UsersSearch, Error500, YandexUID, ServiceMetric, PartnerQuery, RedirectHeatMap,
    UTM, UTM_adwords,  UTM_incoming, UTM_redirect, FlexibleCache, DynamicCache, RivalPosition,
    UserGoalStat, OhmRedirect, OhmUtm, Setting
)
from travel.avia.stat_admin.data.tasks import top_positions_task
from datetimewidget.widgets import DateTimeWidget, DateWidget
from travel.avia.stat_admin.lib.templates import add_common_context

from django.db.models import Avg, Count, Sum
from django import forms
from django.conf import settings
from django.http import JsonResponse, HttpResponseRedirect, HttpResponse, Http404
from django.shortcuts import get_object_or_404, render
from rest_framework.decorators import api_view, authentication_classes, permission_classes
from rest_framework.authentication import BasicAuthentication
from rest_framework.permissions import IsAuthenticated

from travel.avia.stat_admin.lib.jobs import run_task_in_thread
from travel.avia.stat_admin.job.models import ActiveJob
from travel.avia.stat_admin.scripts.cohorts_v2.calc_cohorts import (
    read_data, make_report_data, make_csv_data, make_data_linked_and_prices,
    read_data_with_attr, make_data_linked, make_report_data_decoded, make_csv_data_with_prices
)
from travel.avia.stat_admin.utils.http import ssl_redirect


from travel.avia.stat_admin.scripts.cohorts.connector import parse_xml, save_campaign_prices


log = logging.getLogger(__name__)


AVIA_SERVICES = ['ticket', 'api']
PLATFORM_LEGEND = {
    "501": 'Расписания',
    "502": 'Авиа',
    "503": 'iOS',
    "504": 'Android',
    "505": 'WP',
    "506": 'iPad',
    "507": 'tabled Adroid'
}
CLICK_HOUSE_URL_TEMPLATE = 'http://mtmega.yandex.ru:8123/?query={query}&user={user}&password={password}'
CLICKHOUSE_USER = 'avia'
CLICKHOUSE_PASSWORD = 'QFwsalZM'
MAX_COHORTS_LIMIT = 8


def _utm_to_dict(utm):
    return {
        'source': utm.source,
        'campaign': utm.campaign,
        'medium': utm.medium,
        'term': utm.term,
    },


# Выпилить после RASPTICKETS-9937
@api_view(['GET'])
@authentication_classes((BasicAuthentication,))
@permission_classes((IsAuthenticated,))
@csv_response
def cohorts(request, eventdate, cohorts_num):
    eventdate = datetime.strptime(eventdate, '%Y-%m-%d')
    cohorts_num = int(cohorts_num)

    incoming_data, redirects = read_data(eventdate, eventdate, cohorts_num)
    utms = make_report_data(incoming_data, redirects, cohorts_num)
    header, rows = make_csv_data(utms, cohorts_num)

    data = StringIO()

    csv_writer = csv.writer(data, delimiter=';', quotechar='"', quoting=csv.QUOTE_NONNUMERIC)
    csv_writer.writerow(header)
    csv_writer.writerows(rows)

    return data.getvalue()


# Выпилить после RASPTICKETS-9937
@api_view(['GET'])
@authentication_classes((BasicAuthentication,))
@permission_classes((IsAuthenticated,))
@csv_response
def cohorts_v3_csv(request, eventdate, cohorts_num, whence=None):
    eventdate = datetime.strptime(eventdate, '%Y-%m-%d')
    cohorts_num = int(cohorts_num)

    # Выпилить лишнее после RASPTICKETS-9837
    if whence == 'linked':
        utms = make_data_linked(eventdate, cohorts_num)
    else:
        if whence == "attr":
            incoming_data, redirects = read_data_with_attr(eventdate, eventdate, cohorts_num, version=2)
        else:
            incoming_data, redirects = read_data(eventdate, eventdate, cohorts_num, version=2)

        utms = make_report_data(incoming_data, redirects, cohorts_num)

    header, rows = make_csv_data(utms, cohorts_num)

    data = StringIO()

    csv_writer = csv.writer(data, delimiter=';', quotechar='"', quoting=csv.QUOTE_NONNUMERIC)
    csv_writer.writerow(header)
    csv_writer.writerows(rows)

    return data.getvalue()


def _get_utms(eventdate, cohorts_num, whence):
    if whence == 'linked':
        return make_data_linked_and_prices(eventdate, cohorts_num)

    if whence == "attr":
        incoming_data, redirects = read_data_with_attr(eventdate, eventdate, cohorts_num, version=2)
    else:
        incoming_data, redirects = read_data(eventdate, eventdate, cohorts_num, version=2)

    return make_report_data(incoming_data, redirects, cohorts_num, True)


@api_view(['GET'])
@authentication_classes((BasicAuthentication,))
@permission_classes((IsAuthenticated,))
@csv_response
def cohorts_v5_csv(request, eventdate, cohorts_num, whence=None):
    eventdate = datetime.strptime(eventdate, '%Y-%m-%d')
    cohorts_num = int(cohorts_num)

    utms = _get_utms(eventdate, cohorts_num, whence)
    header, rows = make_csv_data_with_prices(utms, cohorts_num)

    data = StringIO()

    csv_writer = csv.writer(data, delimiter=';', quotechar='"', quoting=csv.QUOTE_NONNUMERIC)
    csv_writer.writerow(header)
    csv_writer.writerows(rows)

    return data.getvalue()


# Выпилить после RASPTICKETS-9937
# Выпилить после RASPTICKETS-9837
def cohorts_v3_decoded(request, eventdate, cohorts_num):
    eventdate = datetime.strptime(eventdate, '%Y-%m-%d')
    cohorts_num = int(cohorts_num)

    incoming_data, redirects = read_data(eventdate, eventdate, cohorts_num, version=2)
    utms = make_report_data_decoded(incoming_data, redirects, cohorts_num)

    investigated_utm = OhmUtm.objects.filter(
        source=request.GET.get('source', ''),
        campaign=request.GET.get('campaign', ''),
        medium=request.GET.get('medium', ''),
        term=request.GET.get('term', '')
    ).first()

    details = []
    error = None
    if not investigated_utm:
        error = 'UTM не найдена'
    else:
        for r in utms[investigated_utm.id]:
            true_utm_str = ','.join([
                r.incoming.utm.source,
                r.incoming.utm.campaign,
                r.incoming.utm.medium,
                r.incoming.utm.term,
            ])

            if len(true_utm_str) > 32:
                true_utm_str = true_utm_str[:32] + '...'

            good_date = r.incoming.eventdate == eventdate.date()
            good_utm = r.incoming.utm.id == investigated_utm.id

            record = {
                'redirect': r,
                'true_utm_str': true_utm_str,
                'good_date': good_date,
                'good_utm': good_utm,
                'bgcolor': '#eeFFee' if good_date and good_utm else '#FFeeEE'
            }

            details.append(record)

    context = {
        'details': details,
        'error': error
    }

    add_common_context(request, context)

    return render(request, 'data/cohorts_details.html', context)


# Выпилить после RASPTICKETS-9937
@api_view(['GET'])
@authentication_classes((BasicAuthentication,))
@permission_classes((IsAuthenticated,))
def cohorts_v3_json(request, eventdate, cohorts_num, whence=None):
    # TODO: Пережимать CSV в JSON - криво.
    # TODO: Но на момент запуска нового API иначе поступать страшно :(
    eventdate = datetime.strptime(eventdate, '%Y-%m-%d')
    cohorts_num = int(cohorts_num)

    if cohorts_num > MAX_COHORTS_LIMIT:
        cohorts_num = MAX_COHORTS_LIMIT

    # Выпилить лишнее после RASPTICKETS-9837
    if whence == 'linked':
        utms = make_data_linked(eventdate, cohorts_num)
    else:
        if whence == "attr":
            incoming_data, redirects = read_data_with_attr(eventdate, eventdate, cohorts_num, version=2)
        else:
            incoming_data, redirects = read_data(eventdate, eventdate, cohorts_num, version=2)

        utms = make_report_data(incoming_data, redirects, cohorts_num)

    header, rows = make_csv_data(utms, cohorts_num)

    data = []
    for row in rows:
        # [u'source', u'campaign', u'medium', u'term', u'total'] + [cohorts]
        cohorts = row[5:]

        item = {
            'utm': {
                'source': row[0],
                'campaign': row[1],
                'medium': row[2],
                'term': row[3],
            },
            'cohort': {x: int(c) for x, c in enumerate(cohorts)},
        }

        data.append(item)

    response_data = {
        'response': data,
        'count': len(data),
        'last_updated': {
            'incomings': Setting.objects.get(key='UTM_V3_LAST_IMPORT_TIME').value,
            'redirects': Setting.objects.get(key='UTM_V3_REDIR_LAST_IMPORT_TIME').value,
        }
    }

    return JsonResponse(response_data)


@api_view(['GET'])
@authentication_classes((BasicAuthentication,))
@permission_classes((IsAuthenticated,))
def cohorts_v5_json(request, eventdate, cohorts_num, whence=None):
    eventdate = datetime.strptime(eventdate, '%Y-%m-%d')
    cohorts_num = int(cohorts_num)

    utms = _get_utms(eventdate, cohorts_num, whence)

    cached_utms = {int(u.id): _utm_to_dict(u) for u in OhmUtm.objects.filter(id__in=utms.keys())}
    data = [
        {
            'utm': cached_utms[utm_id],
            'cohort': {
                c: {
                    'count': item['count'],
                    'price': float(item['price'])
                }
                for c, item in cohort_info.iteritems()
            }
        }
        for utm_id, cohort_info in utms.iteritems()
    ]

    response_data = {
        'response': data,
        'count': len(data),
        'last_updated': {
            'incomings': Setting.objects.get(key='UTM_V3_LAST_IMPORT_TIME').value,
            'redirects': Setting.objects.get(key='UTM_V3_REDIR_LAST_IMPORT_TIME').value,
        }
    }

    return JsonResponse(response_data)


@api_view(['GET'])
@authentication_classes((BasicAuthentication,))
@permission_classes((IsAuthenticated,))
def cohorts_v4_json(request, eventdate, cohorts_num):
    eventdate = datetime.strptime(eventdate, '%Y-%m-%d')
    cohorts_num = int(cohorts_num)

    if cohorts_num > MAX_COHORTS_LIMIT:
        cohorts_num = MAX_COHORTS_LIMIT

    # После RASPTICKETS-9937 заменить make_csv_data на make_json_data
    # тем самым убрать конвертацию csv -> json
    utms = make_data_linked(eventdate, cohorts_num, version=3)
    header, rows = make_csv_data(utms, cohorts_num)

    data = []
    for row in rows:
        # [u'source', u'campaign', u'medium', u'term', u'total'] + [cohorts]
        cohorts = row[5:]

        item = {
            'utm': {
                'source': row[0],
                'campaign': row[1],
                'medium': row[2],
                'term': row[3],
            },
            'cohort': {x: int(c) for x, c in enumerate(cohorts)},
        }

        data.append(item)

    response_data = {
        'response': data,
        'count': len(data),
        'last_updated': {
            'incomings': Setting.objects.get(key='UTM_V4_LAST_IMPORT_TIME').value,
            'redirects': Setting.objects.get(key='UTM_V4_REDIR_LAST_IMPORT_TIME').value,
        }
    }

    return JsonResponse(response_data)


def lost_goals_details(request, yandex_uid, left_date, right_date):
    sql_template = """SELECT
    EventTime,
    URL,
    Params
FROM hits_all
WHERE (EventDate >= '{left_date}') AND (EventDate <= '{right_date}') AND (CounterID = 13979182) AND (UserID = {yandex_uid})"""
    sql = sql_template.format(
        left_date=left_date,
        right_date=right_date,
        yandex_uid=yandex_uid
    )

    re.sub(r'\s+', ' ', sql)

    url = CLICK_HOUSE_URL_TEMPLATE.format(
        query=sql,
        user=CLICKHOUSE_USER,
        password=CLICKHOUSE_PASSWORD,
    )
    r = requests.get(url, timeout=60)

    context = {
        'details': r.content,
    }

    add_common_context(request, context)

    return render(request, 'data/lost_details.html', context)


def compare_metrika_and_we(request, eventdate):
    eventdate = datetime.strptime(eventdate, "%Y-%m-%d")
    right_redirect_date = eventdate + timedelta(days=7)

    sql_template = """select
    UserID,
    firstStartTime,
    firstUTMSource,
    firstUTMCampaign,
    StartDate,
    TraficSourceID,
    SearchEngineID,
    AdvEngineID,
    conversions
from
(
    select
        UserID,
        min(StartTime) as firstStartTime,
        argMin(UTMSource, StartTime) as firstUTMSource,
        argMin(UTMCampaign, StartTime) as firstUTMCampaign
    from
        visits_all
    where
        StartDate >= toDate('{eventdate}') and
        StartDate <= toDate('{eventdate}') and
        CounterID = 13979182 and
        IsRobot = 0
    group by UserID
    HAVING substring(firstUTMSource, 1, 3) = 'ohm'
)
all inner join
(
    select
        UserID,
        StartDate,
        TraficSourceID,
        SearchEngineID,
        AdvEngineID,
        sum(countEqual(Goals.ID, 12787481)*Sign) as conversions
    from
        visits_all
    where
        StartDate >= toDate('{eventdate}') and
        StartDate <= toDate('{right_redirect_date}') and
        CounterID = 13979182 and
        IsRobot = 0 and
        has(Goals.ID, 12787481)
    group by
        UserID,
        StartDate,
        TraficSourceID,
        SearchEngineID,
        AdvEngineID
)
using UserID
"""

    sql = sql_template.format(
        eventdate=eventdate.strftime('%Y-%m-%d'),
        right_redirect_date=right_redirect_date.strftime('%Y-%m-%d'),
    )

    re.sub(r'\s+', ' ', sql)

    url = CLICK_HOUSE_URL_TEMPLATE.format(
        query=sql,
        user=CLICKHOUSE_USER,
        password=CLICKHOUSE_PASSWORD,
    )
    r = requests.get(url, timeout=60)

    compare_data = defaultdict(lambda : defaultdict(int))
    totals = defaultdict(int)

    for line in r.content.strip().split('\n'):
        line = line.strip()
        line_parths = line.split('\t')

        yandexuid = line_parths[0]
        conversions = int(line_parths[8])

        trafic_source_id = int(line_parths[5])
        start_date = line_parths[4]

        # https://st.yandex-team.ru/CMANALYTICS-55#1505144385000
        if trafic_source_id == 3 and start_date != eventdate.strftime('%Y-%m-%d'):
            continue

        compare_data[yandexuid]['goals'] += conversions
        totals['goals'] += conversions

    redirects = OhmRedirect.objects.filter(
        version=2,
        eventdate__gte=eventdate,
        eventdate__lt=eventdate + timedelta(days=7),
        incoming__utm__source__startswith='ohm',
        incoming__eventdate=eventdate,
    ).select_related(
        'incoming'
    )

    for rd in redirects:
        compare_data[rd.yandexuid]['redirects'] += 1
        totals['redirects'] += 1

    for k, v in compare_data.iteritems():
        if v.get('goals') == v.get('redirects'):
            compare_data[k]['bgcolor'] = '#eeFFee'
        else:
            compare_data[k]['bgcolor'] = '#FFeeee'

    context = {
        'compare_data': dict(compare_data),
        'totals': dict(totals),
        'eventdate': eventdate,
    }

    add_common_context(request, context)

    return render(request, 'data/compare_metrika_and_we.html', context)


def compare_metrika_and_we_detailed(request, eventdate, yandexuid):
    eventdate = datetime.strptime(eventdate, "%Y-%m-%d")

    sql_template = """
    SELECT
        UserID,
        StartTime,
        UTMSource,
        UTMCampaign,
        countEqual(Goals.ID, 12787481)
    FROM visits_all
    WHERE (StartDate >= toDate('{eventdate}')) AND
          (StartDate <= toDate('{eventdate}')) AND
          has(Goals.ID, 12787481) AND
          (CounterID = 13979182) AND
          (substring(UTMSource, 1, 3) = 'ohm')
          AND (UserID = {yandexuid})"""

    sql = sql_template.format(
        eventdate=eventdate.strftime('%Y-%m-%d'),
        yandexuid=yandexuid
    )

    re.sub(r'\s+', ' ', sql)

    url = CLICK_HOUSE_URL_TEMPLATE.format(
        query=sql,
        user=CLICKHOUSE_USER,
        password=CLICKHOUSE_PASSWORD,
    )
    r = requests.get(url, timeout=60)

    metrika = []
    for line in r.content.strip().split('\n'):
        line = line.strip()
        line_parths = line.split('\t')

        data = {
            'user_id': line_parths[0],
            'start_time': line_parths[1],
            'utm_source': line_parths[2],
            'utm_campaign': line_parths[3],
            'goals': line_parths[4],
        }
        metrika.append(data)

    redirects = OhmRedirect.objects.filter(
        version=2,
        yandexuid=yandexuid,
        eventdate__gte=eventdate,
        eventdate__lt=eventdate + timedelta(days=7),
        incoming__utm__source__startswith='ohm',
        incoming__eventdate=eventdate,
    ).select_related(
        'incoming'
    )

    avia = []
    for r in redirects:
        data = {
            'eventtime': r.eventdatetime,
            'incoming': {
                'eventtime': r.incoming.eventdatetime,
                'utm_source': r.incoming.utm.source,
                'utm_campaign': r.incoming.utm.campaign,
            }
        }
        avia.append(data)

    context = {
        'metrika': metrika,
        'avia': avia,
        'eventdate': eventdate,
        'yandexuid': yandexuid,
    }

    add_common_context(request, context)

    return render(request, 'data/compare_metrika_and_we_detailed.html', context)


@csv_response
def lost_raw(request, eventdate):
    avia_cache = {}
    redirects = OhmRedirect.objects.filter(eventdate=eventdate).values('yandexuid')
    for r in redirects:
        try:
            uid = int(r['yandexuid'])
        except ValueError:
            continue

        avia_cache[uid] = True

    log.info(avia_cache)

    sql_template = """SELECT
    UserID,
    EventTime,
    dictGetString('UserAgent', 'value', toUInt64(UserAgent)) as UserAgentName,
    ClientIP,
    RegionID,
    IsMobile,
    IsRobot
FROM hits_all
WHERE (EventDate = '{eventdate}') AND (CounterID = 13979182) AND has(GoalsReached, 12787481)"""

    sql = sql_template.format(
        eventdate=eventdate,
    )

    re.sub(r'\s+', ' ', sql)

    url = CLICK_HOUSE_URL_TEMPLATE.format(
        query=sql,
        user=CLICKHOUSE_USER,
        password=CLICKHOUSE_PASSWORD,
    )
    r = requests.get(url, timeout=60)

    data = StringIO()
    csv_writer = csv.writer(data, delimiter=';', quotechar='"', quoting=csv.QUOTE_NONNUMERIC)

    header = ['UserID', 'EventTime', 'UserAgentName', 'ClientIP', 'RegionID', 'IsMobile', 'IsRobot']
    csv_writer.writerow(header)

    if not avia_cache:
        return data.getvalue()

    for line in r.content.strip().split('\n'):
        fields = line.split('\t')
        if avia_cache.get(int(fields[0])):
            continue

        fields[3] = Int2IP(int(fields[3]))
        csv_writer.writerow(fields)

    return data.getvalue()


def Int2IP(ipnum):
    o1 = int(ipnum / 16777216) % 256
    o2 = int(ipnum / 65536) % 256
    o3 = int(ipnum / 256) % 256
    o4 = int(ipnum) % 256
    return '%(o1)s.%(o2)s.%(o3)s.%(o4)s' % locals()


def lost_goals(request):
    left_date = datetime.now().date() - timedelta(days=1)
    right_date = left_date

    if request.method == 'POST':
        form = RivalForm(request.POST)
        if form.is_valid():
            left_date = form.cleaned_data['left_date']
            right_date = form.cleaned_data['right_date']
    else:
        form = RivalForm(
            initial={
            'left_date': left_date,
            'right_date': right_date,
            }
        )

    goal_stat = UserGoalStat.objects.filter(
        eventdate__gte=left_date,
        eventdate__lte=right_date,
        avia_count=0,
    ).values('yandexuid', 'user_agent').annotate(goals_sum=Sum('goals_count')).order_by('-goals_sum')

    redirects_count = OhmRedirect.objects.filter(
        eventdate__gte=left_date,
        eventdate__lte=right_date,
    ).count()

    context = {
        'form': form,
        'left_date': left_date.strftime('%Y-%m-%d'),
        'right_date': right_date.strftime('%Y-%m-%d'),
        'goal_stat': goal_stat,
        'yandexuid_total': len(goal_stat),
        'redirects_count': redirects_count,
        'lost_percent': float(len(goal_stat)) / redirects_count * 100 if redirects_count else 0,
        'lost_money': 23 * len(goal_stat),
    }

    add_common_context(request, context)

    return render(request, 'data/lost.html', context)


def rivals(request):
    right_date = datetime.now().date()
    left_date = right_date - timedelta(days=7)

    if request.method == 'POST':
        form = RivalForm(request.POST)
        if form.is_valid():
            left_date = form.cleaned_data['left_date']
            right_date = form.cleaned_data['right_date']
    else:
        form = RivalForm(
            initial={
            'left_date': left_date,
            'right_date': right_date,
            }
        )

    rivals_short = RivalPosition.objects.filter(
        eventdate__range=[left_date, right_date]
    ).values(
        'rival_name'
    ).annotate(
        position=Avg('position')
    ).order_by(
        'position'
    )

    if right_date != left_date:
        date_range = '%s - %s' % (left_date.strftime('%Y-%m-%d'), right_date.strftime('%Y-%m-%d'))
    else:
        date_range = left_date.strftime('%Y-%m-%d')

    rivals_full = RivalPosition.objects.filter(
        eventdate__range=[left_date, right_date]
    ).values(
        'eventdate', 'rival_name', 'direction', 'passengers', 'one_way', 'partner', 'avia_company', 'minimal_price'
    ).annotate(
        position=Avg('position')
    ).order_by(
        'direction', 'eventdate', 'one_way', 'passengers', 'position'
    )

    # Посчитаем кол-во и список самых дешевых
    rivals_cheaper = RivalPosition.objects.filter(
        eventdate__range=[left_date, right_date],
        cheaper=True
    )

    cheaper = defaultdict(int)
    for rc in rivals_cheaper:
        key = (rc.rival_name, rc.partner)
        cheaper[key] +=1

    cheaper_list = []
    for key, count in cheaper.items():
        rival_name, partner = key
        cheaper_list.append([count, rival_name, partner])

    cheaper_list = sorted(cheaper_list, reverse=True)

    rival_names = RivalPosition.objects.all().order_by('rival_name').values_list('rival_name', flat=True).distinct()

    rivals = RivalPosition.objects.filter(
        eventdate__range=[left_date, right_date]
    ).values(
        'eventdate', 'rival_name',
    ).order_by(
        'eventdate', 'rival_name'
    ).distinct()

    rival_names_dict = defaultdict(set)
    for r in rivals:
        rival_names_dict[r['eventdate']].add(r['rival_name'])

    rival_names_count = []

    for k, v in rival_names_dict.items():
        rival_names_count.append({
            'eventdate': k,
            'count': len(v),
            'missing': [r for r in rival_names if r not in v] or '-'
        })

    context = {
        'form': form,
        'daterange': date_range,
        'rivals_short': rivals_short,
        'rivals_full': rivals_full,
        'rivals_count': rival_names_count,
        'rivals_cheaper': cheaper_list,
    }

    add_common_context(request, context)

    return render(request, 'data/rivals.html', context)


@csv_response
def redir_csv(request, user_redirects):
    data = StringIO()

    csv_writer = csv.writer(data, delimiter=';', quotechar='"', quoting=csv.QUOTE_NONNUMERIC)
    csv_writer.writerow(['UTC datetime', 'count'])

    for r in user_redirects:
        date_time = r['eventdate'].strftime("%d.%m.%Y %H:%M")
        count = int(r['count'])
        headers = [date_time, count]
        csv_writer.writerow(headers)

    return data.getvalue()


@jsonp_response
def redir_json(request, user_redirects):
    json_content = []
    for r in user_redirects:
        json_content.append([int(calendar.timegm(r['eventdate'].timetuple())) * 1000, int(r['count'])])

    return json_content


def need_approve_page(request):
    context = {}

    if settings.YANDEX_ENVIRONMENT_TYPE == 'production':
        idm_host = 'idm.yandex-team.ru'
    else:
        idm_host = 'idm.test.yandex-team.ru'

    context['idm_url'] = 'https://{host}/system/avia-stat#main=roles,roleRequestForm=user:{user};system:avia-stat'.format(
        host=idm_host,
        user=request.user.username
    )

    add_common_context(request, context)

    return render(request, 'data/need_approve.html', context)


def redirects(request):
    period = request.GET.get('period', '')
    output_format = request.GET.get('format', 'json')
    service = request.GET.get('service')
    national_version = request.GET.get('national_version')
    pp = request.GET.get('pp')

    if service:
        services = [service]
    else:
        services = AVIA_SERVICES

    try:
        limit = int(request.GET.get('limit', ''))

    except Exception:
        limit = 0

    model_filter = {'service__in': services}

    if national_version:
        model_filter['national_version'] = national_version

    if pp:
        model_filter['pp'] = pp

    if period == 'hour':
        left_border = datetime.now() - timedelta(hours=limit)
        model_filter['eventdate__gte'] = left_border
        user_redirects = UserRedirect.objects.filter(**model_filter).values('eventdate').annotate(count=Sum('count')).order_by('-eventdate')

    elif period == 'day':
        left_border = datetime.now() - timedelta(days=limit)
        model_filter['eventdate__gte'] = left_border
        model_filter['eventdate__lte'] = datetime.utcnow().date()
        user_redirects = UserRedirect.objects.filter(**model_filter).extra({'eventdate': 'DATE(eventdate)'}).values('eventdate').annotate(count=Sum('count')).order_by('-eventdate')
    elif period == 'month':
        model_filter['eventdate__gte'] = datetime(2014, 10, 1, 4, 0, 0)
        model_filter['eventdate__lte'] = datetime.utcnow().date()

        user_redirects = UserRedirect.objects.filter(**model_filter).extra({'eventdate': 'DATE(CONCAT(extract(YEAR from eventdate), \'-\', extract(MONTH from eventdate), \'-\', 1))'}).values('eventdate').annotate(count=Sum('count')).order_by('-eventdate')[:limit]  # noqa

    if output_format == 'csv':
        output_content = redir_csv(request, user_redirects)

    else:
        output_content = redir_json(request, user_redirects)

    add_common_context(request, output_content)

    return output_content


@jsonp_response
def search_json(request, user_search):
    json_content = []
    for r in user_search:
        json_content.append([int(calendar.timegm(r['eventdate'].timetuple())) * 1000, int(r['count'])])

    return json_content


def search(request):
    period = request.GET.get('period', '')
    output_format = request.GET.get('format', 'json')
    service = request.GET.get('service')
    national_version = request.GET.get('national_version')
    pp = request.GET.get('pp')

    if service:
        services = [service]
    else:
        services = AVIA_SERVICES

    model_filter = {'service__in': services}

    if national_version:
        model_filter['national_version'] = national_version

    if pp:
        model_filter['pp'] = pp

    try:
        limit = int(request.GET.get('limit', ''))

    except Exception:
        limit = 0

    if period == 'hour':
        left_border = datetime.now() - timedelta(hours=limit)
        user_search = UsersSearch.objects.filter(
            eventdate__gte=left_border,
            **model_filter
        ).extra({'eventdate': 'eventdate'}).values('eventdate').annotate(count=Sum('count')).order_by('-eventdate')

    elif period == 'day':
        left_border = datetime.now() - timedelta(days=limit)
        model_filter['eventdate__gte'] = left_border
        model_filter['eventdate__lte'] = datetime.utcnow().date()

        user_search = UsersSearch.objects.filter(**model_filter).extra({'eventdate': 'DATE(eventdate)'}).values('eventdate').annotate(count=Sum('count')).order_by('-eventdate')

    elif period == 'month':
        user_search = UsersSearch.objects.filter(
            eventdate__lte=datetime.utcnow().date(),
            **model_filter
        ).extra({'eventdate': 'DATE(CONCAT(extract(YEAR from eventdate), \'-\', extract(MONTH from eventdate), \'-\', 1))'}).values('eventdate').annotate(count=Sum('count')).order_by('-eventdate')[:limit]  # noqa

    else:
        return {}

    if output_format == 'csv':
        # FIXME! csv
        output_content = redir_csv(request, user_search)

    else:
        output_content = search_json(request, user_search)

    add_common_context(request, output_content)

    return output_content


def daemon(request):
    context = {}
    left_date_border = date.today() - timedelta(days=14)

    daemon_query_time_avgs = ServiceMetric.objects.filter(
        event_name='daemon_query_time_avg',
        event_datetime__gte=left_date_border,
    )

    chart_data1 = defaultdict(list)
    for daemon_query_time_avg in daemon_query_time_avgs:
        chart_data1[str(daemon_query_time_avg.service)].append([
            calendar.timegm(daemon_query_time_avg.event_datetime.timetuple()) * 1000,
            int(daemon_query_time_avg.value)
        ])

    context['daemon_query_time_avg'] = dict(chart_data1)

    daemon_variants_len_avgs = ServiceMetric.objects.filter(
        event_name='daemon_variants_len_avg',
        event_datetime__gte=left_date_border,
    )

    chart_data2 = defaultdict(list)
    for daemon_variants_len_avg in daemon_variants_len_avgs:
        chart_data2[str(daemon_variants_len_avg.service)].append([
            calendar.timegm(daemon_variants_len_avg.event_datetime.timetuple()) * 1000,
            int(daemon_variants_len_avg.value)
        ])

    context['daemon_variants_len_avg'] = dict(chart_data2)

    daemon_requests_partners_avg = ServiceMetric.objects.filter(
        event_name='daemon_requests_partners_avg',
        event_datetime__gte=left_date_border,
    )

    chart_data3 = defaultdict(list)
    for daemon_requests_partners_avg in daemon_requests_partners_avg:
        chart_data3[str(daemon_requests_partners_avg.service)].append([
            calendar.timegm(daemon_requests_partners_avg.event_datetime.timetuple()) * 1000,
            int(daemon_requests_partners_avg.value)
        ])

    context['daemon_requests_partners_avg'] = dict(chart_data3)

    cache_hits = ServiceMetric.objects.filter(
        event_name='cache_hit',
        event_datetime__gte=left_date_border,
    )

    chart_data4 = defaultdict(list)
    for cache_hit in cache_hits:
        chart_data4[str(cache_hit.service)].append([
            calendar.timegm(cache_hit.event_datetime.timetuple()) * 1000,
            int(cache_hit.value)
        ])

    context['cache_hit'] = dict(chart_data4)

    add_common_context(request, context)

    return render(request, 'data/need_approve.html', context)


def index(request):
    context = {}
    yesterday = (datetime.now() - timedelta(days=1)).date()
    totay = datetime.now().date()

    yesterday_clicks = UserRedirect.objects.filter(
        national_version='ru',
        service__in=AVIA_SERVICES,
        eventdate__range=(yesterday, totay)
    ).aggregate(
        Sum('count')
    ).get('count__sum', 0)

    yesterday_searches = UsersSearch.objects.filter(
        national_version='ru',
        service__in=AVIA_SERVICES,
        eventdate__range=(yesterday, totay)
    ).aggregate(
        Sum('count')
    ).get('count__sum', 0)

    yesterday_errors = Error500.objects.filter(
        eventdate=yesterday
    ).aggregate(
        Sum('count')
    ).get('count__sum', 0)

    try:
        yesterday_visitors = YandexUID.objects.get(
            eventdate=yesterday,
            service__in=AVIA_SERVICES,
            national_version='ru',
        ).count

    except YandexUID.DoesNotExist:
        yesterday_visitors = 0

    context['yesterday'] = {
        'date': yesterday,
        'visitors': yesterday_visitors,
        'clicks': yesterday_clicks if yesterday_clicks else 0,
        'searches': yesterday_searches if yesterday_searches else 0,
        'errors': yesterday_errors if yesterday_errors else 0,
    }

    add_common_context(request, context)

    return render(request, 'data/index.html', context)


def ping(request):
    return HttpResponse("I live...", content_type='text/plain')


def version(request):
    from django.conf import settings

    return HttpResponse(
        '{}'.format(settings.PKG_VERSION),
        content_type='text/plain'
    )


def round_chartdata(cd, tiks=100, min_datetime=None, max_datetime=None):
    if not cd:
        return [], min_datetime, max_datetime

    if len(cd) < tiks:
        tiks = len(cd)

    if not min_datetime:
        min_datetime = min(cd)[0]

    if not max_datetime:
        max_datetime = max(cd)[0]

    timerange = (max_datetime - min_datetime).total_seconds()
    timeseek = int(float(timerange) / tiks)

    chart_data_slices = defaultdict(list)

    for ed, qs in cd:
        idx = int((ed - min_datetime).total_seconds() / timeseek)
        chart_data_slices[idx].append(qs)

    chart_data = []
    for k, v in chart_data_slices.items():
        chart_data.append([
            calendar.timegm((min_datetime + timedelta(seconds=k*timeseek)).timetuple()) * 1000,
            int(sum(v) / len(v))
        ])

    return chart_data, min_datetime, max_datetime


def clicks(request):
    national_version = request.GET.get('national_version', 'ru')
    left_date = datetime.now() - timedelta(days=7)
    searches = RedirectHeatMap.objects.filter(
        eventdate__gte=left_date.date(),
        national_version=national_version
    )
    data = [[s.lat, s.lon] for s in searches]

    context = {}
    context['data'] = data

    add_common_context(request, context)

    return render(request, 'data/online_clicks.html', context)


def connector_upload(request):
    if request.method == 'POST':
        form = ConnectorFileForm(request.POST, request.FILES)
        if form.is_valid():
            xml_content = request.FILES['file'].read()
            campaigns_iter = parse_xml(xml_content)
            save_campaign_prices(campaigns_iter)

    return HttpResponseRedirect('/online/cohort/')


@csv_response
def cohort_csv(request, results):
    data = StringIO()

    csv_writer = csv.writer(data, delimiter=';', quotechar='"', quoting=csv.QUOTE_NONNUMERIC)
    csv_writer.writerow(['campaign', 'incoming', 'outgoing', 'expenses', 'income', 'profit', 'incoming adw', 'income adw', 'profit adw', '0', '1', '2', '3', '4', '5', '6', '7', '8'])

    for campaign, c_data in results.items():
        headers = [
            campaign,
            c_data.get('inc', 0) if c_data else '-',
            c_data.get('rec', 0) if c_data else '-',
            c_data.get('inc_price', 0) if c_data else '-',
            c_data.get('rec_price', 0) if c_data else '-',
            c_data.get('price_delta', 0) if c_data else '-',
            c_data.get('adw', 0) if c_data else '-',
            c_data.get('adw_price', 0) if c_data else '-',
            c_data.get('adw_price_delta', 0) if c_data else '-',
            c_data.get(0, 0) if c_data else '-',
            c_data.get(1, 0) if c_data else '-',
            c_data.get(2, 0) if c_data else '-',
            c_data.get(3, 0) if c_data else '-',
            c_data.get(4, 0) if c_data else '-',
            c_data.get(5, 0) if c_data else '-',
            c_data.get(6, 0) if c_data else '-',
            c_data.get(7, 0) if c_data else '-',
            c_data.get(8, 0) if c_data else '-'
        ]
        csv_writer.writerow(headers)

    return data.getvalue()


def cohort(request):
    context = {}

    if request.method == 'POST':
        form = CohortForm(request.POST)

        if form.is_valid():
            source = form.cleaned_data['source']
            incoming_start = form.cleaned_data['incoming_start']
            incoming_end = form.cleaned_data['incoming_end']

            results = defaultdict(str)

            db_filter = {
                'utm__source': source,
                'eventdate__range': (incoming_start, incoming_end),
            }
            for campaign in UTM_incoming.objects.filter(**db_filter).values('utm__campaign').annotate(total=Count('utm__campaign'), price=Sum('price')):
                if campaign['utm__campaign'] not in results:
                    price_delta = campaign['price'] * -1

                    results[campaign['utm__campaign']] = {
                        'inc': campaign['total'],
                        'inc_price': campaign['price'],
                        'rec': 0,
                        'rec_price': 0,
                        'price_delta': price_delta,
                        'adw': 0,
                        'adw_price': 0,
                        'adw_price_delta': 0,
                        'td_bgcolor': 'd8bfd8' if price_delta < 0 else '',
                        'adw_td_bgcolor': '',
                    }

                    for x in range(12):
                        results[campaign['utm__campaign']][x] = 0

            db_filter['eventdate__range'] = (incoming_start, incoming_end)
            for campaign in UTM_redirect.objects.filter(**db_filter).values('utm__campaign', 'cohort').annotate(total=Count('utm__campaign'), price=Sum('price')):
                try:
                    if int(campaign['cohort']) < 9:
                        results[campaign['utm__campaign']][campaign['cohort']] = campaign['total']
                        results[campaign['utm__campaign']]['rec'] += campaign['total']
                        results[campaign['utm__campaign']]['rec_price'] += settings.AVIA_CLICK_PRICE * campaign['total']

                        price_delta = results[campaign['utm__campaign']]['rec_price'] - results[campaign['utm__campaign']]['inc_price']
                        results[campaign['utm__campaign']]['price_delta'] = price_delta

                        if price_delta < 0:
                            results[campaign['utm__campaign']]['td_bgcolor'] = 'd8bfd8'
                        elif price_delta > 0:
                            results[campaign['utm__campaign']]['td_bgcolor'] = 'c0d6be'
                except Exception:
                    log.exception("ERROR")
                    continue

            for campaign in UTM_adwords.objects.filter(**db_filter).values('utm__campaign').annotate(total=Sum('clicks'), price=Sum('cost')):
                try:
                    results[campaign['utm__campaign']]['adw'] += campaign['total']
                    results[campaign['utm__campaign']]['adw_price'] += campaign['price']

                    adw_price_delta = results[campaign['utm__campaign']]['rec_price'] - results[campaign['utm__campaign']]['adw_price']
                    results[campaign['utm__campaign']]['adw_price_delta'] = adw_price_delta

                    if adw_price_delta < 0:
                        results[campaign['utm__campaign']]['adw_td_bgcolor'] = 'd8bfd8'
                    elif adw_price_delta > 0:
                        results[campaign['utm__campaign']]['adw_td_bgcolor'] = 'c0d6be'

                except Exception:
                    log.exception("ERROR")
                    continue

            if form.cleaned_data['csv']:
                return cohort_csv(request, results)

            context['results'] = dict(results)
            context['source'] = source
    else:
        yesterday = date.today() - timedelta(days=1)
        left_border = date.today() - timedelta(days=8 * 7)

        form = CohortForm(initial={
            'source': ([('google', 'google'), ('serp', 'serp'), ('ya', 'ya'), ('yamain', 'yamain'), ('yamap', 'yamap')]),
            'incoming_start': left_border,
            'incoming_end': yesterday,
        })

    context['form'] = form
    context['connector_form'] = ConnectorFileForm()

    add_common_context(request, context)

    return render(request, 'data/cohort.html', context)


def visitors(request):
    context = {}
    yesterday = (datetime.now() - timedelta(days=1)).date()

    context['errors'] = errors
    context['yesterday'] = yesterday

    yandexuids = YandexUID.objects.all().order_by('-eventdate')[:30]
    chart_data = []
    table_data = []

    for yandexuid in yandexuids:
        chart_data.append([
            calendar.timegm(yandexuid.eventdate.timetuple()) * 1000,
            int(yandexuid.count)
        ])

        table_data.append([
            yandexuid.eventdate,
            int(yandexuid.count)
        ])

    context['chart_data'] = chart_data
    context['table_data'] = yandexuids
    context['yesterday'] = {
        'date': yesterday
    }

    add_common_context(request, context)

    return render(request, 'data/online_visitors.html', context)


def partner_queries(request):
    context = {}
    good_statuses = ['got_reply']

    if request.method == 'POST':
        form = PartnerForm(request.POST)
        if form.is_valid():
            partner_code = form.cleaned_data['partner']
            start = form.cleaned_data['start']
            end = form.cleaned_data['end']
    else:
        start = datetime.combine((datetime.now() - timedelta(days=2)).date(), datetime.min.time())
        end = datetime.now()
        form = PartnerForm(initial={
            'start': start,
            'end': end,
            'partner': {'m': 'male'}
        })
        partner_code = 'ALL'

    query_filter = {
        'query_datetime__gte': start.replace(second=0, microsecond=0),
        'query_datetime__lte': end.replace(second=0, microsecond=0),
    }

    if partner_code != 'ALL':
        query_filter['partner_code'] = partner_code

    status_filter = {
        'query_status__in': good_statuses
    }

    good_queries = PartnerQuery.objects.filter(**dict(query_filter.items() + status_filter.items())).values('query_datetime').annotate(count_sum=Sum('count'), query_time_avg=Avg('avg_query_time'))

    status_filter = {'query_status__in': good_statuses + ['query_invalid', 'got_empty_reply']}
    bad_queries = PartnerQuery.objects.filter(**query_filter).exclude(**status_filter).values('query_datetime').annotate(count_sum=Sum('count'))
    status_filter = {'query_status': 'query_invalid'}
    query_invalid = PartnerQuery.objects.filter(**dict(query_filter.items() + status_filter.items())).values('query_datetime').annotate(count_sum=Sum('count'))
    status_filter = {'query_status': 'got_empty_reply'}
    empty_query = PartnerQuery.objects.filter(**dict(query_filter.items() + status_filter.items())).values('query_datetime').annotate(count_sum=Sum('count'))
    status_filter = {'query_status': 'timeout'}
    timeout_query = PartnerQuery.objects.filter(**dict(query_filter.items() + status_filter.items())).values('query_datetime').annotate(count_sum=Sum('count'))
    status_filter = {'query_status': 'response_error'}
    response_error = PartnerQuery.objects.filter(**dict(query_filter.items() + status_filter.items())).values('query_datetime').annotate(count_sum=Sum('count'))

    good_data = []
    bad_data = []
    query_invalid_data = []
    empty_query_data = []
    timeout_query_data = []
    query_time_avg_data = []
    response_error_data = []

    for pq in good_queries:
        good_data.append([
            pq['query_datetime'] + timedelta(hours=3),
            int(pq['count_sum'] / 10)
        ])

    for pq in good_queries:
        query_time_avg_data.append([
            pq['query_datetime'] + timedelta(hours=3),
            int(pq['query_time_avg'] / 10)
        ])

    for pq in response_error:
        response_error_data.append([
            pq['query_datetime'] + timedelta(hours=3),
            int(pq['count_sum'] / 10)
        ])

    for pq in bad_queries:
        bad_data.append([
            pq['query_datetime'] + timedelta(hours=3),
            int(pq['count_sum'] / 10)
        ])

    for pq in query_invalid:
        query_invalid_data.append([
            pq['query_datetime'] + timedelta(hours=3),
            int(pq['count_sum'] / 10)
        ])

    for pq in empty_query:
        empty_query_data.append([
            pq['query_datetime'] + timedelta(hours=3),
            int(pq['count_sum'] / 10)
        ])

    for pq in timeout_query:
        timeout_query_data.append([
            pq['query_datetime'] + timedelta(hours=3),
            int(pq['count_sum'] / 10)
        ])

    if good_data:
        good_queries, min_datetime, max_datetime = round_chartdata(good_data, tiks=100)
        bad_queries, _, _ = round_chartdata(bad_data, tiks=100, min_datetime=min_datetime, max_datetime=max_datetime)
        query_invalid, _, _ = round_chartdata(query_invalid_data, tiks=100, min_datetime=min_datetime, max_datetime=max_datetime)
        empty_query, _, _ = round_chartdata(empty_query_data, tiks=100, min_datetime=min_datetime, max_datetime=max_datetime)
        timeout_query, _, _ = round_chartdata(timeout_query_data, tiks=100, min_datetime=min_datetime, max_datetime=max_datetime)
        query_time_avg, _, _ = round_chartdata(query_time_avg_data, tiks=100, min_datetime=min_datetime, max_datetime=max_datetime)
        response_error, _, _ = round_chartdata(response_error_data, tiks=100, min_datetime=min_datetime, max_datetime=max_datetime)
    else:
        good_queries = []
        bad_queries = []
        query_invalid = []
        empty_query = []
        timeout_query = []
        query_time_avg = []
        response_error = []

    bad_queries_dict = {k: v for k, v in bad_queries}

    bad_queries_processed = []
    for dt, val in good_queries:
        if bad_queries_dict.get(dt):
            all_queries = float(bad_queries_dict.get(dt)) + float(val)
            good_percent = float(val) / all_queries * 100
        else:
            good_percent = 100

        bad_queries_processed.append([dt, good_percent])

    context['good_queries'] = good_queries
    context['bad_queries'] = bad_queries
    context['good_percent'] = bad_queries_processed
    context['query_invalid'] = query_invalid
    context['empty_query'] = empty_query
    context['timeout_query'] = timeout_query
    context['query_time_avg'] = query_time_avg
    context['response_error'] = response_error

    context['form'] = form

    add_common_context(request, context)

    return render(request, 'data/partner_queries.html', context)


def top100(request):
    context = {}

    return render(request, 'data/report_top100.html', context)


def errors(request):
    context = {}
    yesterday = (datetime.now() - timedelta(days=1)).date()

    errors = Error500.objects.filter(
        eventdate=yesterday
    )

    statuses = {
        500: 'Internal Server Error',
        501: 'Not Implemented',
        502: 'Bad Gateway',
        503: 'Service Unavailable',
        504: 'Gateway Timeout',
        505: 'HTTP Version Not Supported',
        506: 'Variant Also Negotiates',
        507: 'Insufficient Storage',
        508: 'Loop Detected',
        509: 'Bandwidth Limit Exceeded',
        510: 'Not Extended',
        511: 'Network Authentication Required',
    }

    context['errors'] = errors
    context['statuses'] = statuses
    context['yesterday'] = {
        'date': yesterday
    }

    errors = Error500.objects.filter(
        eventdate__gte=(yesterday - timedelta(days=14)),
        eventdate__lte=yesterday,
    ).values(
        'eventdate'
    ).annotate(
        count_sum=Sum('count')
    ).order_by(
        'eventdate'
    )

    chart_data = []

    for error in errors:
        chart_data.append([
            calendar.timegm(error['eventdate'].timetuple()) * 1000,
            int(error['count_sum'])
        ])

    context['chart_data'] = chart_data

    add_common_context(request, context)

    return render(request, 'data/online_errors.html', context)


def searches(request):
    yesterday = (datetime.now() - timedelta(days=1)).date()
    if request.method == 'POST':
        form = SearchAndRedirectsForm(request.POST)
        if form.is_valid():
            national_version = form.cleaned_data['national_version']
            service = form.cleaned_data['service']
            pp = form.cleaned_data['pp']

    else:
        form = SearchAndRedirectsForm()
        national_version = ''
        service = ''
        pp = ''

    # Подсказка для заголовка
    filter_legend = []
    if service:
        filter_legend.append(service)
    if national_version:
        filter_legend.append(national_version)
    if pp:
        filter_legend.append(pp)

    context = {
        'form': form,
        'yesterday': {'date': yesterday},
        'national_version': national_version,
        'service': service,
        'pp': pp,
        'filter_legend': filter_legend,
    }

    add_common_context(request, context)

    return render(request, 'data/online_searches.html', context)


def job_download(request, job_id):
    job = get_object_or_404(ActiveJob, id=job_id)
    zip_filename ='avia-stats-%s.zip' % job_id

    if not job.binary:
        raise Http404("No data")

    resp = HttpResponse(job.binary, content_type="application/x-zip-compressed")
    resp['Content-Disposition'] = 'attachment; filename=%s' % zip_filename

    return resp


def jobs(request):
    yesterday = (datetime.now() - timedelta(days=1)).date()

    context = {}
    context['yesterday'] = {
        'date': yesterday,
    }
    context['jobs'] = get_jobs(request.user)

    add_common_context(request, context)

    return render(request, 'data/jobs.html', context)


def get_jobs(user):
    jobs = ActiveJob.objects.filter(
        user_id=user.id,
    ).order_by(
        '-id'
    )

    return jobs


def top_positions(request):
    if request.method == 'POST':
        form = TopNumericForm(request.POST)

        if form.is_valid():
            partner_code = form.cleaned_data['partner']

            date_from = form.cleaned_data['start']
            date_to = form.cleaned_data['end']

            params = {
                'start_date': date_from,
                'end_date': date_to,
                'mailto': form.cleaned_data['email'].split('@')[0],
                'national_version': form.cleaned_data['national_version'],
                'quantity': form.cleaned_data['quantity'],
            }

            if partner_code != 'ALL':
                params['partner'] = partner_code

            job_title = u'Топ 100 и средняя позиция в выдаче за %s - %s' % (
                date_from.strftime('%d.%m.%Y'),
                date_to.strftime('%d.%m.%Y')
            )

            new_job = ActiveJob(
                user=request.user,
                title=job_title,
                state='Starting',
                script='top_positions_task',
                params=', '.join(['%s' % k if not v else '%s: %s' % (k, v) for k, v in params.items()]),
            )

            new_job.save()

            params['job'] = str(new_job.id)

            top_positions_task.apply_async((params, new_job))

            return ssl_redirect(request, '/jobs/')

    else:
        now = datetime.now().date()
        form = TopNumericForm(initial={
            'email': request.user.email,
            'start': date(now.year, now.month, 1),
            'end': now,
            'partner': {'m': 'male'},
            'quantity': 100,
        })

    context = {
        'form': form,
        'status': request.GET.get('status')
    }
    add_common_context(request, context)

    return render(request, 'data/report_top100.html', context)


def partner_query_time(request):
    if request.method == 'POST':
        form = TopForm(request.POST)

        if form.is_valid():
            partner_code = form.cleaned_data['partner']

            script = 'query_time'

            date_from = form.cleaned_data['start']
            date_to = form.cleaned_data['end']

            params = {
                '-s': date_from.strftime('%Y-%m-%d'),
                '-e': date_to.strftime('%Y-%m-%d'),
                '-m': form.cleaned_data['email'].split('@')[0],
                '-n': form.cleaned_data['national_version'],
            }

            if partner_code != 'ALL':
                params['-p'] = partner_code

            run_task_in_thread(
                user=request.user,
                title=u'Время ответов %s - %s' % (
                    date_from.strftime('%d.%m.%Y'),
                    date_to.strftime('%d.%m.%Y')
                ),
                script=script,
                params=params
            )

            return ssl_redirect(request, '/jobs/')

    else:
        now = datetime.now().date()
        form = TopForm(initial={
            'email': request.user.email,
            'start': date(now.year, now.month, 1),
            'end': now,
            'partner': {'m': 'male'},
        })

    context = {
        'form': form,
        'status': request.GET.get('status')
    }
    add_common_context(request, context)

    return render(request, 'data/partner_queri_time.html', context)


def partners(request):
    if request.method == 'POST':
        form = TopForm(request.POST)

        if form.is_valid():
            partner_code = form.cleaned_data['partner']

            script = 'partners_requests'

            date_from = form.cleaned_data['start']
            date_to = form.cleaned_data['end']

            params = {
                '-s': date_from.strftime('%Y-%m-%d'),
                '-e': date_to.strftime('%Y-%m-%d'),
                '-m': form.cleaned_data['email'].split('@')[0],
                '-n': form.cleaned_data['national_version'],
            }

            if partner_code != 'ALL':
                params['-p'] = partner_code

            run_task_in_thread(
                user=request.user,
                title=u'Запросы к парнерам за %s - %s' % (
                    date_from.strftime('%d.%m.%Y'),
                    date_to.strftime('%d.%m.%Y')
                ),
                script=script,
                params=params
            )

            return ssl_redirect(request, '/jobs/')

    else:
        now = datetime.now().date()
        form = TopForm(initial={
            'email': request.user.email,
            'start': date(now.year, now.month, 1),
            'end': now,
            'partner': {'m': 'male'}
        })

    context = {
        'form': form,
        'status': request.GET.get('status')
    }
    add_common_context(request, context)

    return render(request, 'data/partners.html', context)


def returnings(request):
    if request.method == 'POST':
        form = TopForm(request.POST)

        if form.is_valid():
            partner_code = form.cleaned_data['partner']

            script = 'returning'

            date_from = form.cleaned_data['start']
            date_to = form.cleaned_data['end']

            params = {
                '-s': date_from.strftime('%Y-%m-%d'),
                '-e': date_to.strftime('%Y-%m-%d'),
                '-m': form.cleaned_data['email'].split('@')[0],
            }

            if partner_code != 'ALL':
                params['-p'] = partner_code

            run_task_in_thread(
                user=request.user,
                title=u'Возвраты за %s - %s' % (
                    date_from.strftime('%d.%m.%Y'),
                    date_to.strftime('%d.%m.%Y')
                ),
                script=script,
                params=params
            )

            return ssl_redirect(request, '/jobs/')

    else:
        now = datetime.now().date()
        form = TopForm(initial={
            'email': request.user.email,
            'start': date(now.year, now.month, 1),
            'end': now,
            'partner': {'m': 'male'}
        })

    context = {
        'form': form,
        'status': request.GET.get('status')
    }
    add_common_context(request, context)

    return render(request, 'data/returnings.html', context)


def partner_redirects(request):
    if request.method == 'POST':
        form = TopForm(request.POST)

        if form.is_valid():
            partner_code = form.cleaned_data['partner']

            script = 'redirect'

            date_from = form.cleaned_data['start']
            date_to = form.cleaned_data['end']

            params = {
                '-s': date_from.strftime('%Y-%m-%d'),
                '-e': date_to.strftime('%Y-%m-%d'),
                '-m': form.cleaned_data['email'].split('@')[0],
            }

            if partner_code != 'ALL':
                params['-p'] = partner_code

            run_task_in_thread(
                user=request.user,
                title=u'Редиректы за %s - %s' % (
                    date_from.strftime('%d.%m.%Y'),
                    date_to.strftime('%d.%m.%Y')
                ),
                script=script,
                params=params
            )

            return ssl_redirect(request, '/jobs/')

    else:
        now = datetime.now().date()
        form = TopForm(initial={
            'email': request.user.email,
            'start': date(now.year, now.month, 1),
            'end': now,
            'partner': {'m': 'male'}
        })

    context = {
        'form': form,
        'status': request.GET.get('status')
    }
    add_common_context(request, context)

    return render(request, 'data/redirects.html', context)


def get_partners():
    data_sources_map = []
    partners_url = '%s/partner/codes/' % settings.RASP_API_HOST
    vendors_url = '%s/dohop_vendors/codes/' % settings.RASP_API_HOST
    try:
        r = requests.get(
            partners_url,
            verify=False,
            timeout=3,
        )

        for partner in r.json():
            billing_datasource_id_production = partner['billing_datasource_id_production']
            code = partner['code']
            if billing_datasource_id_production:
                data_sources_map.append((code, code))

        r = requests.get(
            vendors_url,
            verify=False,
            timeout=3,
        )

        for partner in r.json():
            title = 'dohop: %s' % partner['title']
            code = partner['code']
            if partner['enabled']:
                data_sources_map.append((code, title))

    except Exception:
        data_sources_map = {}

    return (('ALL', u'Все'), ) + tuple(sorted(data_sources_map))


def flexiblecache_table_view(request):
    context = {}

    flexible_context_30 = defaultdict(list)
    flexible_context_90 = defaultdict(list)
    flexible_context = defaultdict(list)

    left_border_30_days = datetime.now().date() - timedelta(days=30)
    left_border_90_days = datetime.now().date() - timedelta(days=90)

    for f in FlexibleCache.objects.all():
        item = [
            int(calendar.timegm(f.eventdate.timetuple())) * 1000,
            f.percent
        ]
        if f.eventdate >= left_border_30_days:
            flexible_context_30[str(f.national_version)].append(item)

        if f.eventdate >= left_border_90_days:
            flexible_context_90[str(f.national_version)].append(item)

        flexible_context[str(f.national_version)].append(item)

    context['flexible_30'] = dict(flexible_context_30)
    context['flexible_90'] = dict(flexible_context_90)
    context['flexible'] = dict(flexible_context)

    return render(request, 'data/flexible_cache.html', context)


def dynamiccache_table_view(request):
    context = {}
    dynamic_context = defaultdict(list)

    month_year_keys = [(datetime.now() + relativedelta(months=x)).strftime("%Y-%m") for x in range(6)]

    cache = defaultdict(list)

    for f in DynamicCache.objects.filter(month_year__in=month_year_keys):
        if f.national_version not in cache:
            cache[f.national_version] = defaultdict(list)

        cache[f.national_version][f.month_year].append(f.percent)

    for national_version in cache:
        for k, v in cache[national_version].items():
            year, month = k.split('-')
            month_year = date(int(year), int(month), 1)

            dynamic_context[str(national_version)].append([
                int(calendar.timegm(
                    datetime.combine(month_year, datetime.min.time()).timetuple()
                )) * 1000,
                int(float(sum(v)) / len(v))
            ])
        dynamic_context[str(national_version)] = sorted(dynamic_context[str(national_version)])

    context['dynamic'] = dict(dynamic_context)

    return render(request, 'data/dynamic_cache.html', context)


class TopForm(forms.Form):
    email = forms.EmailField(label=u'Ваш email', max_length=100)
    partner = forms.ChoiceField(label=u'Партнер', widget=forms.Select, choices=get_partners())
    start = forms.DateField(label=u'Дата начала', widget=DateWidget(
        attrs={'id': "id_start"}, usel10n=True, bootstrap_version=3,
        options={'format': 'dd.mm.yyyy'}
    ))
    end = forms.DateField(label=u'Дата окончания', widget=DateWidget(
        attrs={'id': "id_end"}, usel10n=True, bootstrap_version=3,
        options={'format': 'dd.mm.yyyy'}
    ))
    national_version = forms.ChoiceField(label=u'Нац. версия', widget=forms.Select,
                                         choices=tuple([(n, n) for n in settings.AVIA_NATIONAL_VERSIONS]))


class TopNumericForm(TopForm):
    quantity = forms.IntegerField(label=u'Количество направлений', initial=100,
                                  min_value=1, max_value=100000)


class PartnerForm(forms.Form):
    partner = forms.ChoiceField(label=u'Партнер', widget=forms.Select, choices=get_partners())
    start = forms.DateTimeField(label=u'Дата начала', widget=DateTimeWidget(
        attrs={'id': "id_start"},
        usel10n=True,
        bootstrap_version=3,
        options={
            'format': 'dd.mm.yyyy hh:ii',
        }
    ))
    end = forms.DateTimeField(label=u'Дата окончания', widget=DateTimeWidget(
        attrs={'id': "id_end"},
        usel10n=True,
        bootstrap_version=3,
        options={
            'format': 'dd.mm.yyyy hh:ii',
        }
    ))


class CohortForm(forms.Form):
    source = forms.ChoiceField(
        label=u'Источник',
        choices=([(s['source'], s['source']) for s in UTM.objects.values('source').distinct()]),
        initial='3',
        required=True,
    )
    incoming_start = forms.DateField(label=u'Входящие дата начала', widget=DateWidget(
        attrs={'id': 'id_incoming_start'}, usel10n=True, bootstrap_version=3,
        options={'format': 'dd.mm.yyyy'}
    ))
    incoming_end = forms.DateField(label=u'Входящие дата окончания', widget=DateWidget(
        attrs={'id': 'id_incoming_end'}, usel10n=True, bootstrap_version=3,
        options={'format': 'dd.mm.yyyy'},
    ))

    csv = forms.BooleanField(label=u'CSV', required=False)


class ConnectorFileForm(forms.Form):
    file = forms.FileField()


class SearchAndRedirectsForm(forms.Form):
    service_chooices = sorted([(r['service'], r['service']) for r in UserRedirect.objects.values('service').distinct()])
    national_chooices = sorted([(r['national_version'], r['national_version']) for r in UserRedirect.objects.values('national_version').distinct()])
    pp_chooices = sorted([(r['pp'], PLATFORM_LEGEND.get(r['pp'], r['pp'])) for r in UserRedirect.objects.values('pp').distinct()])

    service = forms.ChoiceField(
        label=u'Сервис', widget=forms.Select,
        choices=[('', 'Все')] + service_chooices,
        required=False,
    )

    national_version = forms.ChoiceField(
        label=u'Национальная версия', widget=forms.Select,
        choices=[('', 'Все')] + national_chooices,
        required=False,
    )

    pp = forms.ChoiceField(
        label=u'Платформа', widget=forms.Select,
        choices=[('', 'Все')] + pp_chooices,
        required=False,
    )


class RivalForm(forms.Form):
    left_date = forms.DateField(label=u'Дата начала', widget=DateWidget(
        attrs={'id': 'id_incoming_start'}, usel10n=True, bootstrap_version=3,
        options={'format': 'dd.mm.yyyy'}
    ))
    right_date = forms.DateField(label=u'Дата окончания', widget=DateWidget(
        attrs={'id': 'id_incoming_end'}, usel10n=True, bootstrap_version=3,
        options={'format': 'dd.mm.yyyy'},
    ))
