from logging import Logger

import logging
from django.db import transaction
from typing import List

from travel.avia.stat_admin.data.models.settlement import Settlement
from travel.avia.stat_admin.data.models.airport import Airport


class TitleModelImporter(object):
    def __init__(self, logger, Model):
        # type: (Logger) -> None
        self._logger = logger
        self._Model = Model

    def import_models(self, raw_models):
        # type: (List[dict]) -> dict

        self._logger.info(
            'start stats export %s',
            self._Model.__class__.__name__
        )
        models = list(self._Model.objects.all())
        self._logger.info(
            'finish export stats %s. Object count %d',
            self._Model.__class__.__name__, len(models)
        )

        existing_models_ids = {s.avia_id for s in models}
        models_ids = {s['id'] for s in raw_models}
        id_to_raw_model = {s['id']: s for s in raw_models}
        id_to_existing_model = {s.avia_id: s for s in models}

        new_models_ids = models_ids - existing_models_ids
        changed_models_ids = {
            model_id
            for model_id in models_ids - new_models_ids
            if (id_to_raw_model[model_id]['title'] !=
                id_to_existing_model[model_id].title)
        }

        changed_models_ids = list(changed_models_ids)[:1000]
        new_models_ids = list(new_models_ids)[:1000]

        with transaction.atomic():
            for model_id in changed_models_ids:
                model = id_to_existing_model[model_id]
                model.title = id_to_raw_model[model_id]['title']
                model.save()
            for model_id in new_models_ids:
                raw_model = id_to_raw_model[model_id]
                self._Model.objects.create(
                    avia_id=model_id,
                    title=raw_model['title']
                )

        return {
            'new': len(new_models_ids),
            'changed': len(changed_models_ids),
            'ignored': (
                len(models_ids)
                - len(new_models_ids)
                - len(changed_models_ids)
            )
        }

settlement_importer = TitleModelImporter(
    logger=logging.getLogger(__name__ + '.settlement'),
    Model=Settlement
)
airport_importer = TitleModelImporter(
    logger=logging.getLogger(__name__ + '.airport'),
    Model=Airport
)
