# -*- coding: utf-8 -*-
import errno
import os
import logging
import pickle
from functools import wraps


def debug_cache(cache, log=None, enable=True, reset=False):
    if log is None:
        log = logging.getLogger(__name__)
    _reset = [reset]
    def cacher(fn):
        if not enable:
            return fn
        @wraps(fn)
        def wrapper(*args, **kwargs):
            try:
                if _reset[0]:
                    _reset[0] = False
                    raise IOError('Reset cache')
                result = cache.load_result()
                log.debug('Using result from cache: %s' % cache)
            except (IOError, OSError, pickle.PickleError):
                result = fn(*args, **kwargs)
                cache.save_result(result)
                log.debug('New result saved to cache: %s' % cache)
            return result
        return wrapper
    return cacher


class FileCache(object):
    def __init__(self, filepath):
        self._filepath = filepath

    def load_result(self):
        with open(self._filepath, 'rb') as f:
            return pickle.load(f)

    def save_result(self, data):
        ensure_dir(os.path.dirname(self._filepath))
        with open(self._filepath, 'wb') as f:
            pickle.dump(data, f)

    def __str__(self):
        return 'FileCache %r' % self._filepath


def ensure_dir(dirpath):
    try:
        os.makedirs(dirpath)
    except OSError as ex:
        if ex.errno != errno.EEXIST:
            raise
