# -*- encoding: utf-8 -*-
# По аналогии с логами из админки
import inspect
import logging
import os
import sys
import traceback
from copy import copy

from django.conf import settings

from travel.avia.stat_admin.lib.exceptions import SimpleUnicodeException
from travel.avia.stat_admin.lib.fileutils import get_my_caller_file


LOG_FORMAT = '%(name)s %(process)d %(asctime)s %(levelname)s: %(message)s'


class LogError(SimpleUnicodeException):
    pass


def add_stdout_handler(log, format=None):
    format = format or LOG_FORMAT
    """Добавляем handler для stdout, если еще не добавлен"""
    # Проверяем есть ли у нас handler который пишет в stdout
    add_stream_handler(log, sys.stdout, format)


def print_log_to_stdout(log=''):
    add_stdout_handler(log)


def add_stream_handler(log, stream, format=None, filters=None):
    """Добавляем handler для stream, если еще не добавлен"""

    format = format or LOG_FORMAT

    log = _get_log_from_string_or_log_instance(log)
    filters = filters or []

    for handler in log.handlers:
        if isinstance(handler, logging.StreamHandler) and handler.stream == stream:
            return

    stream_handler = logging.StreamHandler(stream)
    stream_handler.setFormatter(ContextRaspFormatter(format))

    for log_filter in filters:
        stream_handler.addFilter(log_filter)

    log.addHandler(stream_handler)

    return stream_handler


def _get_log_from_string_or_log_instance(log_or_logname):
    if isinstance(log_or_logname, basestring) or log_or_logname is None:
        return logging.getLogger(log_or_logname)
    else:
        return log_or_logname


class ContextRaspFormatter(logging.Formatter):
    def __init__(self, fmt=None, datefmt=None):
        if fmt:
            self._simple_fmt = fmt
        else:
            self._simple_fmt = "%(message)s"

        if ':' in self._simple_fmt:
            self._context_fmt = '%s: %%(context)s:%s' % tuple(self._simple_fmt.split(':', 1))
        else:
            self._context_fmt = self._simple_fmt + ' : ' + '%(context)s'

        self.datefmt = datefmt

    def formatException(self, ei):
        """
        Новая реализация
        """
        exc_info = traceback.format_exception(ei[0], ei[1], ei[2])
        exc_info = "".join(exc_info).strip()
        if not isinstance(exc_info, unicode):
            exc_info = str(exc_info).decode('utf8', 'replace')
        return exc_info

    def format(self, record):
        if hasattr(record, 'context') and record.context:
            self._fmt = self._context_fmt
        else:
            self._fmt = self._simple_fmt

        if record.name == '__main__' and getattr(record, 'main_file_path', None):
            record = copy(record)
            record.name = record.main_file_path

        return logging.Formatter.format(self, record)


# Позволяет генерировать лог на основе места вызова функции
def get_current_file_logger_name():
    stack = inspect.stack()
    caller_file = get_my_caller_file(stack)
    return get_file_logger_name(caller_file)


def get_file_logger_name(py_file):
    """
    Получаем название лога для файла проекта
    """
    if not py_file.endswith(u'.py'):
        raise LogError(u"{} не является файлом проекта".format(py_file))

    file_path = os.path.abspath(py_file)
    project_path = os.path.abspath(settings.PROJECT_PATH)

    if file_path.startswith(project_path):
        relative_path = file_path.replace(project_path, u'', 1).lstrip(os.path.sep)
        logger_name = relative_path.replace(os.path.sep, u'.')
        logger_name = u'.'.join(logger_name.split(u'.')[:-1])  # remove .py(pyc) extension

        parts = logger_name.split(u'.')
        if parts[-1] == '__init__':
            logger_name = u'.'.join(parts[:-1])

        return logger_name
    else:
        raise LogError(u"%s не является файлом проекта не смогли создать лог" % file_path)


def create_current_file_run_log(format=None, capture_stdstreams=False):
    stack = inspect.stack()
    caller_file = get_my_caller_file(stack)

    log_name = 'special.script_runs.' + get_file_logger_name(caller_file)

    return create_run_log(log_name, format, capture_stdstreams=capture_stdstreams)


def create_run_log(log_name, format=None, capture_stdstreams=False):
    from datetime import datetime
    from django.conf import settings
    format = format or settings.LOG_FORMAT

    path = os.path.join(settings.LOG_PATH, *log_name.split('.'))
    path = os.path.join(path, datetime.now().strftime('%Y-%m-%d_%H%M%S.log'))

    if not os.path.exists(os.path.dirname(path)):
        os.makedirs(os.path.dirname(path))

    handler = logging.FileHandler(path)
    handler.setFormatter(ContextRaspFormatter(format))

    log = logging.getLogger()
    log.addHandler(handler)

    if capture_stdstreams:
        capture_stdstreams_to_file(handler.stream)

    return handler.stream


def capture_stdstreams_to_file(file):
    """
    Перенаправляем стандартные потоки в этот файл
    """
    fd = file.fileno()

    sys.stdout.flush()
    os.close(1)
    os.dup2(fd, 1)

    sys.stderr.flush()
    os.close(2)
    os.dup2(fd, 2)
