# -*- coding: utf-8 -*-
import logging
import re
import tempfile

import boto3
from django.conf import settings

log = logging.getLogger(__name__)


class S3SyncError(Exception):
    pass


def bucket_exists(s3, buck):
    buckets = list(s3.buckets.all())
    return buck in buckets


def get_bucket():
    endpoint_url = settings.MDS_S3_ENDPOINT_URL
    bucket_name = settings.MDS_S3_REVISE_SCREENSHOTS_BUCKET

    s3 = boto3.resource('s3', endpoint_url=endpoint_url)
    bucket = s3.Bucket(bucket_name)

    try:
        if not bucket_exists(s3, bucket):
            log.info('Bucket does not exist, creating new bucket %s',
                     bucket_name)
            s3.create_bucket(
                Bucket=bucket_name,
                CreateBucketConfiguration={
                    'LocationConstraint': 'us-east-1',
                },
            )
        else:
            log.debug('Bucket %s was found in storage', bucket)
        return bucket
    except Exception as e:
        msg = (
            'Connection failed (%s). Please check for correct S3 credentials in environment variables '
            '(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY) for host %s'
        )
        log.exception(msg, e, endpoint_url)
        raise S3SyncError(msg % (e, endpoint_url))


def upload_binary(bucket, data, s3_key, content_type):
    try:
        with tempfile.NamedTemporaryFile() as tf:
            with open(tf.name, 'wb') as fh:
                fh.write(data)

            bucket.Object(s3_key).upload_file(
                tf.name,
                ExtraArgs={
                    'ContentType': content_type,
                }
            )

        return 'https://{bucket_name}.{s3_domain}/{s3_key}'.format(
            bucket_name=bucket.name,
            s3_domain=re.sub(r'https?://', '', settings.MDS_S3_ENDPOINT_URL),
            s3_key=s3_key
        )
    except Exception as e:
        msg = 'Uploading failed (%s). s3_key=%s bucket=%s'
        log.exception(msg, e, s3_key, bucket)
        raise S3SyncError(msg % (e, s3_key, bucket))
