# -*- coding: utf-8 -*-
from contextlib import contextmanager
from datetime import date, datetime, timedelta

from django.conf import settings

import yt.wrapper as yt


@contextmanager
def temp_table(yt):
    table = yt.create_temp_table(settings.YT_TMP_DIRECTORY)
    try:
        yield table
    finally:
        yt.remove(table)


def configure_wrapper(yt):
    yt.config['proxy']['url'] = settings.YT_PROXY
    yt.config['token'] = settings.YT_TOKEN
    yt.config['read_retries']['enable'] = True
    yt.config['clear_local_temp_files'] = True
    yt.config["remote_temp_tables_directory"] = settings.YT_TMP_DIRECTORY


def last_logs_tables(yt, path, days):
    today = datetime.now().date()
    return tables_for_daterange(yt, path, today-timedelta(days), today)


def _table_date(table):
    return datetime.strptime(table.split('/')[-1], '%Y-%m-%d').date()


def tables_for_daterange(yt, path, begin, end):
    return [
        t for t in yt.search(path, node_type='table')
        if begin <= _table_date(t) <= end
    ]


def yt_last_logs_tables(path, days, skip_today=False):
    tables = []
    today = date.today()

    for table in yt.search(path, node_type="table"):
        try:
            table_date = datetime.strptime(table.split('/')[-1], '%Y-%m-%d').date()
        except ValueError:
            continue

        if skip_today and table_date == today:
            continue

        tables.append((
            table_date,
            table
        ))

    sorded_tables = sorted(tables, reverse=True)[:days]

    return [table for _table_date, table in sorded_tables]
