# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import logging

from blackbox import FIELD_LOGIN, BlackboxError
from django.conf import settings
from django.contrib.auth.models import User
from rest_framework import authentication, exceptions

from travel.library.python.tvm_ticket_provider import provider_fabric

from travel.avia.stat_admin.middleware.auth import blackbox

log = logging.getLogger(__name__)


client = provider_fabric.create(settings)


class YandexOAuthAuthentication(authentication.BaseAuthentication):
    def authenticate(self, request):
        authorization = request.META.get('HTTP_AUTHORIZATION', None)
        # http://www.django-rest-framework.org/api-guide/authentication/#custom-authentication
        # If authentication is not attempted, return None. Any other
        # authentication schemes also in use will still be checked.
        if not authorization:
            return None

        userip = request.META.get('REMOTE_ADDR') or request.META.get('HTTP_X_REAL_IP')
        ya_login = get_blackbox_oauth(authorization, userip)
        if not ya_login:
            raise exceptions.AuthenticationFailed('Bad authorization')

        try:
            user = User.objects.get(username=ya_login)
        except User.DoesNotExist:
            raise exceptions.AuthenticationFailed('No such user with login: {}'.format(ya_login))

        return (user, None)

    # If the .authenticate_header() method is not overridden, the
    # authentication scheme will return HTTP 403 Forbidden responses when an
    # unauthenticated request is denied access.
    def authenticate_header(self, request):
        return 'OAuth <oauth_token>'


def get_blackbox_oauth(authorization, userip):
    log.debug('get_blackbox_oauth %r, %r', authorization, userip)
    if not blackbox:
        raise BlackboxError('No blackbox')

    headers = {'Authorization': authorization}
    ticket = client.get_ticket(settings.BLACKBOX_TVM_NAME)
    if ticket:
        headers['X-Ya-Service-Ticket'] = ticket
    try:
        data = blackbox.oauth(
            headers_or_token=headers,
            userip=userip,
            dbfields=[FIELD_LOGIN],
            by_token=False,
        )

    except BlackboxError as exc:
        log.error('Blackbox: %s', exc)
        return None

    log.debug('Blackbox user data: %r', data)

    if data.get('valid', False):
        return data['fields']['login']

    return None


class Unauthorized(Exception):
    pass
