import os

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'travel.avia.stat_admin.local_settings')
import django
django.setup()

import argparse
import logging
from datetime import datetime, timedelta

import pytz
from django.db import transaction

from travel.avia.stat_admin.data.models import PartnerQuery, ArchivePartnerQuery
from travel.avia.stat_admin.lib.logs import add_stdout_handler, create_current_file_run_log


LEFT_BORDER = datetime.today() - timedelta(days=2 * 365)


logger = logging.getLogger(__name__)


def _parse_date(date_str):
    return datetime.strptime(date_str, '%Y-%m-%d')


@transaction.atomic()
def move(to_move):
    new_models = [
        ArchivePartnerQuery(
            id=pq.id,
            service=pq.service,
            query_datetime=pq.query_datetime,
            query_status=pq.query_status,
            partner_code=pq.partner_code,
            count=pq.count,
            avg_query_time=pq.avg_query_time,
        )
        for pq in to_move
    ]

    ArchivePartnerQuery.objects.bulk_create(new_models)
    PartnerQuery.objects.filter(pk__in=[x.pk for x in to_move]).delete()


def main():
    create_current_file_run_log()

    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true', dest='verbose')
    parser.add_argument('--batch-size', type=int, default=5000)
    parser.add_argument('--before', type=_parse_date, default=LEFT_BORDER)
    args = parser.parse_args()

    if args.verbose:
        add_stdout_handler(logger)

    try:
        logger.info('Start')
        left_border = pytz.timezone('Europe/Moscow').localize(args.before)
        logger.info('Archive all borders before %s', left_border.strftime('%Y-%m-%dT%H:%M:%S'))

        while True:
            to_move = PartnerQuery.objects.filter(
                query_datetime__lte=left_border
            )[:args.batch_size]
            if not to_move:
                break

            move(to_move)
            logger.info('Moved %d', len(to_move))

        logger.info('End')
    except:
        logger.exception('ERROR')
        raise


if __name__ == '__main__':
    main()
