# -*- encoding: utf-8 -*-
import os
import sys

from copy import copy
from datetime import datetime
from dateutil.relativedelta import relativedelta
from optparse import Option, OptionParser, OptionValueError

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger


def check_number(option, opt, value):
    try:
        days = int(value)

    except ValueError:
        raise OptionValueError(
            "option %s: bad number: %r" % (opt, value))

    return days


def main():
    os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'travel.avia.stat_admin.local_settings')
    import django
    django.setup()

    import logging
    from django.conf import settings
    from django.db import transaction

    from travel.avia.stat_admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.stat_admin.lib.yt_helpers import configure_wrapper, yt_last_logs_tables
    from travel.avia.stat_admin.data.models import FlexibleCache

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    def map_table(record):
        national_version = record['national_version']
        iso_eventtime = datetime.strptime(record['iso_eventtime'], '%Y-%m-%d %H:%M:%S')
        prices_requested = int(record['prices_requested'])
        prices_found = int(record['prices_found'])

        if prices_found:
            hit_percent = int(float(prices_found) / prices_requested * 100)
        else:
            hit_percent = 0

        yield {
            'national_version': national_version,
            'eventdate': iso_eventtime.date().strftime('%Y-%m-%d'),
            'percent': hit_percent,
        }

    def reduce_table(key, records):
        records_count = 0
        percent_sum = 0

        for r in records:
            records_count += 1
            percent_sum += int(r['percent'])

        yield {
            'national_version': key['national_version'],
            'eventdate': key['eventdate'],
            'percent': int(float(percent_sum) / records_count),
            'count': records_count
        }

    def build_flexible_cache(tmp_table, log):
        logs_quantity = options.days
        source_tables = yt_last_logs_tables(
            '//home/rasp/logs/rasp-flexible-cache-log',
            logs_quantity,
            skip_today=True
        )

        log.info('Map tables: %s -> %s', source_tables, tmp_table)
        yt.run_map(
            map_table,
            source_table=source_tables,
            destination_table=tmp_table,
            format=yt.DsvFormat(),
            spec={"data_size_per_job": settings.YT_DATA_SIZE_PER_JOB},
        )

        log.info('Sort table: %s ', tmp_table)
        yt.run_sort(
            source_table=tmp_table,
            sort_by=['national_version', 'eventdate'],
        )

        log.info('Reduce table: %s ', tmp_table)
        yt.run_reduce(
            reduce_table,
            tmp_table,
            tmp_table,
            format=yt.DsvFormat(),
            reduce_by=['national_version', 'eventdate'],
            spec={"data_size_per_job": settings.YT_DATA_SIZE_PER_JOB},
        )

    @transaction.atomic
    def store_flexible_cache(tmp_table, log):
        left_date_border = datetime.now() - relativedelta(years=1)
        log.info('Remove data before %s', left_date_border.strftime('%Y-%m-%d'))
        FlexibleCache.objects.filter(
            eventdate__lte=left_date_border
        ).delete()

        log.info('Read and store flexible cache from %s to DB', tmp_table)
        for record in yt.read_table(tmp_table, format=yt.JsonFormat(), raw=False):
            national_version = record['national_version']
            eventdate = datetime.strptime(record['eventdate'], '%Y-%m-%d')
            percent = int(record['percent'])
            count = int(record['count'])

            flexible_cache, created = FlexibleCache.objects.get_or_create(
                national_version=national_version,
                eventdate=eventdate,
                defaults={
                    'percent': percent,
                    'count': count,
                }
            )

            skip_update = [
                not created,
                int(flexible_cache.count) == int(count),
                int(flexible_cache.percent) == int(percent)
            ]

            if all(skip_update):
                continue

            flexible_cache.count = count
            flexible_cache.percent = percent

            flexible_cache.save()

        return FlexibleCache.objects.all().count()

    # BEGIN main()
    optparser = OptionParser(option_class=Yoption)

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option("-d", "--days", dest="days", type="days", help="number of last logs to aggregate", default=14)

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)
    else:
        yt_logger_config.level = 'ERROR'
        reload(yt_logger)

    log.info('Start')

    try:
        configure_wrapper(yt)

        log.info('Number of days: %s' % options.days)
        log.info('---------------------------------')

        tmp_table_flexible = yt.create_temp_table(
            path=settings.TOP_DIRECTIONS_TMP_PATH,
            prefix='rasp_flexible_'
        )

        build_flexible_cache(tmp_table_flexible, log)
        rows_stored = store_flexible_cache(tmp_table_flexible, log)

        log.info('%s rows in database', rows_stored)

        yt.remove(tmp_table_flexible)

    except Exception:
        log.exception('Error:')
        sys.exit(1)

    log.info('Done')


class Yoption(Option):
    TYPES = Option.TYPES + ("days", "quantity", "countries")
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
    TYPE_CHECKER["days"] = check_number
    TYPE_CHECKER["quantity"] = check_number


if __name__ == '__main__':
    main()
