# -*- encoding: utf-8 -*-
import os

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'travel.avia.stat_admin.local_settings')
import django
django.setup()

import logging
import errno
import signal

from collections import defaultdict
from datetime import date, datetime, timedelta
from functools import wraps
from optparse import OptionParser

from django.conf import settings
from django.db import transaction
from django.db.models import Count

from travel.avia.stat_admin.data.models import UTM, UTM_adwords, UTM_incoming, UTM_redirect, UTM_ReportsHistory
from adwords_report.report import AdwordsReport

log = logging.getLogger(__name__)

AVIA_CLICK_PRICE = 23  # с НДС
MAX_JUNK_COUNT = 3

ADWORDS_SECONDS_TTL = 15 * 60
ADWORDS_MAX_RETRY = 3


class TimeoutError(Exception):
    pass


def timeout(seconds, error_message=os.strerror(errno.ETIME)):
    def decorator(func):
        def _handle_timeout(signum, frame):
            raise TimeoutError(error_message)

        def wrapper(*args, **kwargs):
            signal.signal(signal.SIGALRM, _handle_timeout)
            signal.alarm(seconds)
            try:
                result = func(*args, **kwargs)
            finally:
                signal.alarm(0)
            return result

        return wraps(func)(wrapper)

    return decorator


def report_history_sucess(googleads_path, report_date):
    try:
        UTM_ReportsHistory.objects.get(
            report_date=report_date,
            yaml_file=os.path.basename(googleads_path),
            success=True
        )
    except UTM_ReportsHistory.DoesNotExist:
        return False

    return True


@timeout(ADWORDS_SECONDS_TTL)
def get_adwords(googleads_path, report_date):
    report_date_str = report_date.strftime('%Y%m%d')

    result = AdwordsReport(googleads_path).get_campaing_utm_params(report_date_str, report_date_str)
    log.info("Received %s report items", len(result))

    aggr_results = defaultdict(lambda : defaultdict(int))

    for r in result:
        utm_params = r["utm_params"]

        utm_source = utm_params.get("utm_source") or ""
        utm_campaign = utm_params.get("utm_campaign") or ""
        utm_medium = utm_params.get("utm_medium") or "cpc"

        key = (utm_source, utm_campaign, utm_medium)

        aggr_results[key]["cost"] += float(r["cost"]) / 1000000
        aggr_results[key]["clicks"] += float(r["clicks"])

    return result, aggr_results


@transaction.atomic
def store_results(aggr_results, report_date):
    for item, r in aggr_results.items():
        utm_source, utm_campaign, utm_medium = item
        cost = float(r["cost"])
        clicks = r["clicks"]
        avg_cost = cost / clicks if clicks else 0

        log.info("source: %s, campaign: %s, medium: %s, cost: %s, count: %s, avg_cost: %s", utm_source, utm_campaign, utm_medium, cost, clicks, avg_cost)

        try:
            utm = UTM.objects.get(
                source=utm_source,
                campaign=utm_campaign,
                medium=utm_medium
            )

        except UTM.DoesNotExist:
            log.info('DoesNotExist')
            continue

        log.info('Store adwords data')
        adwords_utm, created = UTM_adwords.objects.get_or_create(
            eventdate=report_date,
            utm=utm
        )

        adwords_utm.cost = cost
        adwords_utm.clicks = clicks

        adwords_utm.save()


def get_adwords_report(report_date):
    for googleads_path in settings.GOOGLEADS_PATHS:
        if report_history_sucess(googleads_path, report_date):
            log.info('Allready imported: %s, %s', googleads_path, report_date.strftime('%Y-%m-%d'))
            continue

        result = None
        aggr_results = None

        utm_history, created = UTM_ReportsHistory.objects.get_or_create(
            report_date=report_date,
            yaml_file=os.path.basename(googleads_path),
            defaults={
                'eventdatetime': datetime.now(),
                'success': False,
                'last_error': 'Crash after start'
            }
        )

        for x in range(ADWORDS_MAX_RETRY):
            try:
                log.info("Get report: %s %s", googleads_path, report_date)
                result, aggr_results = get_adwords(googleads_path, report_date)

            except Exception as e:
                message = 'Exception: {exc}. Retry {n_from}/{n_to}'.format(
                    exc=str(e),
                    n_from=x + 1,
                    n_to=ADWORDS_MAX_RETRY
                )
                log.error(message)

                utm_history.success = False
                utm_history.last_error = message
                utm_history.report_content = None

                utm_history.save()

                continue

            break

        if result:
            utm_history.success = True
            utm_history.last_error = None
            utm_history.report_content = result

            utm_history.save()

        if aggr_results:
            store_results(aggr_results, report_date)


def clean_utms(report_date):
    # Удаляем (низкочастотный) мусор
    log.info('Get junk_utm_ids')
    junk_utm_ids = [u['utm'] for u in UTM_incoming.objects.filter(eventdate=report_date).values('utm').annotate(total=Count('utm')).filter(total__lte=MAX_JUNK_COUNT)]

    # Исклюим из мусорных реальные кампании по данным адвордс
    log.info('Get adword_utm_ids')
    adword_utm_ids = set([u["utm_id"] for u in UTM_adwords.objects.filter(eventdate=report_date).values('utm_id')])

    log.info('Make junk_utm_ids')
    junk_utm_ids = [i for i in junk_utm_ids if i not in adword_utm_ids]

    log.info('Clean UTM_incoming')
    UTM_incoming.objects.filter(utm__id__in=junk_utm_ids, eventdate=report_date).delete()
    log.info('Clean UTM_redirect')
    UTM_redirect.objects.filter(utm__id__in=junk_utm_ids, eventdate=report_date).delete()


def main():
    log.info("Start")

    optparser = OptionParser()
    optparser.add_option("-d", "--days", dest="days", type="int", help="number of days", default=1)
    options, args = optparser.parse_args()

    for day in range(options.days):
        report_date = date.today() - timedelta(days=day + 1)
        get_adwords_report(report_date)

        clean_utms(report_date)

    log.info("Done")


if __name__ == "__main__":
    main()
