# -*- encoding: utf-8 -*-
import os

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'travel.avia.stat_admin.local_settings')
import django
django.setup()

import gzip
import io
import logging
import subprocess
import tempfile

from datetime import datetime, timedelta
from email.mime.application import MIMEApplication

from django.conf import settings
from django.core.mail.message import EmailMultiAlternatives

from pytz import timezone

from travel.avia.stat_admin.lib.logs import create_current_file_run_log

log = logging.getLogger(__name__)

MAX_COHORTS = 8
MIN_COHORTS = 0
MOSCOW_TIMEZONE = timezone('Europe/Moscow')
ALLOWED_ENVS = ['production', 'development']

EMAIL_RECIPIENTS = [
    'ohm@ohmsd.com',
    'arina.l@ohmsd.com',
    'puzyrev.d@ohmsd.com',
    'arancio@yandex-team.ru',
    'serj0@yandex-team.ru',
    'zynaps@yandex-team.ru',
    'olegpro@yandex-team.ru',
    'pagein@yandex-team.ru',
]


def make_reports(tmp_dir):
    log.info('Make reposts using {tmp_dir}'.format(tmp_dir=tmp_dir))

    cur_dir = os.path.dirname(os.path.abspath(__file__))
    today = MOSCOW_TIMEZONE.localize(datetime.now()).date()
    last_monday = today - timedelta(days=today.weekday(), weeks=1)

    for x in range(MIN_COHORTS, MAX_COHORTS):
        left_date = last_monday - timedelta(days=x*7)
        cohorts_num = x - MIN_COHORTS + 1

        shell_args = [
            'python',
            '{cur_dir}/calc_cohorts.py'.format(cur_dir=cur_dir),
            '-l', '{left_date}'.format(left_date=left_date.strftime('%Y-%m-%d')),
            '-c', '{cohorts_num}'.format(cohorts_num=cohorts_num),
            '-d', '{tmp_dir}'.format(tmp_dir=tmp_dir),
        ]

        log.info('Exec: %s', shell_args)

        subprocess.check_call(shell_args)


def send_reports(tmp_dir):
    log.info('Send reposts from {tmp_dir}'.format(tmp_dir=tmp_dir))

    report_date = MOSCOW_TIMEZONE.localize(datetime.now())
    email = EmailMultiAlternatives(
        subject=u'Когортный отчет от {report_date}'.format(report_date=report_date.strftime('%d.%m.%Y')),
        body=u'Файлы во вложении:\n\n',
        from_email=settings.SERVER_EMAIL,
        to=EMAIL_RECIPIENTS,
    )

    file_names = filter(os.path.isfile, (
        os.path.join(tmp_dir, fname) for fname in os.listdir(tmp_dir)
    ))

    for file_name in sorted(file_names):
        with open(file_name, 'rb') as f, io.BytesIO() as b:
            g = gzip.GzipFile(mode='wb', fileobj=b, compresslevel=9)
            g.writelines(f)
            g.close()
            attachment = MIMEApplication(b.getvalue(), 'x-gzip')
            base_filename = os.path.basename(file_name)
            attachment['Content-Disposition'] = 'attachment; filename=%s.gz' % base_filename

            email.attach(attachment)

    email.send()


def main():
    create_current_file_run_log()
    log.info('Start')

    current_env = settings.YANDEX_ENVIRONMENT_TYPE

    if current_env not in ALLOWED_ENVS:
        log.info("Environment %s not in %s", current_env, ALLOWED_ENVS)
        return

    try:
        tmp_dir = tempfile.mkdtemp()
        make_reports(tmp_dir)
        send_reports(tmp_dir)
    except Exception:
        log.exception('ERROR')

    log.info('Done')


if __name__ == '__main__':
    main()
