# -*- coding: utf-8 -*-
import os

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'travel.avia.stat_admin.local_settings')
import django
django.setup()

import logging
from datetime import datetime, timedelta
from optparse import OptionParser

from django.db.transaction import atomic

from travel.avia.stat_admin.data.models import OhmIncoming, OhmRedirect, Setting
from travel.avia.stat_admin.scripts.cohorts_v3.import_topics import (
    DATA_VERSION, MOSCOW_TIMEZONE
)

from travel.avia.stat_admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.stat_admin.lib.yt_helpers import configure_wrapper

import yt.wrapper as yt
import yt.logger as yt_logger
import yt.logger_config as yt_logger_config


log = logging.getLogger(__name__)

MOD_TIME_DTF = '%Y-%m-%d %H:%M:%S+%Z'


def import_redirects(left_date, right_date, check_last_run=True):
    setting_key = 'UTM_V3_REDIR_LAST_IMPORT_TIME'
    setting_defaults = {
        'value': '',
    }

    setting, _ = Setting.objects.get_or_create(key=setting_key, defaults=setting_defaults)

    days = (right_date - left_date).days
    yt_tables = []
    last_modification_time = None

    for td in range(days + 1):
        table = '//home/avia/logs/avia-redir-balance-by-day-log/{date}'.format(
            date=(left_date + timedelta(days=td)).strftime('%Y-%m-%d'),
        )
        if not yt.exists(table):
            log.info('Table [%s] does not exist, abort', table)
            return

        modification_time = yt.get_attribute(table, 'modification_time')

        if not last_modification_time or modification_time > last_modification_time:
            last_modification_time = modification_time

        yt_tables.append(table)

    if check_last_run and setting.value == last_modification_time:
        log.info('No changes in %s', yt_tables)
        return

    log.info(
        'Delete redirects from %s to %s',
        left_date.strftime('%Y-%m-%d'),
        right_date.strftime('%Y-%m-%d'),
    )
    OhmRedirect.objects.filter(
        eventdate__gte=left_date,
        eventdate__lte=right_date,
        version=DATA_VERSION,
    ).delete()

    ohm_redirects = []
    for yt_table in yt_tables:
        for row in yt.read_table(yt_table, format=yt.JsonFormat()):
            if int(row.get('FILTER')) != 0:
                continue
            raw_iso_eventtime = row.get('ISO_EVENTTIME')

            if not raw_iso_eventtime:
                log.warning('Empty iso_eventtime: %r', row)
                continue

            eventtime = MOSCOW_TIMEZONE.localize(
                datetime.strptime(raw_iso_eventtime, '%Y-%m-%d %H:%M:%S')
            )

            if not left_date <= eventtime.date() <= right_date:
                log.warning('Bad redirect date: %r', eventtime.date())
                continue

            yandexuid = row.get('YANDEXUID')

            click_price = row.get('PRICE_CPA')
            if click_price is None:
                click_price = 19.49
            else:
                click_price = float(click_price) / 100

            if yandexuid:
                # TODO: Наверное, можно взять все yandexuid и last incoming запросом?!
                # TODO: Если делать, то отдельным таском
                last_incoming = OhmIncoming.objects.filter(
                    version=DATA_VERSION,
                    yandexuid=yandexuid,
                    eventdatetime__lte=eventtime,
                ).order_by(
                    '-eventdatetime'
                ).first()

                ohm_redirect = OhmRedirect(
                    eventdate=eventtime.date(),
                    eventdatetime=eventtime,
                    yandexuid=yandexuid,
                    version=DATA_VERSION,
                    incoming=last_incoming,
                    click_price=click_price,
                )

                ohm_redirects.append(ohm_redirect)

    if ohm_redirects:
        log.info('Trying to save %d redirects', len(ohm_redirects))
        OhmRedirect.objects.bulk_create(ohm_redirects, batch_size=1000)

    if check_last_run:
        setting.value = last_modification_time
        setting.save()

    log.info('%r redirects saved to db', len(ohm_redirects))


@atomic
def main():
    optparser = OptionParser()

    default_right_date = datetime.now().strftime('%Y-%m-%d')
    default_left_date = (datetime.now() - timedelta(days=1)).strftime('%Y-%m-%d')

    optparser.add_option('-l', '--left_date', dest='left_date', default=default_left_date)
    optparser.add_option('-r', '--right_date', dest='right_date', default=default_right_date)
    optparser.add_option('--check_last_run', action='store_true')
    optparser.add_option('-v', '--verbose', action='store_true')

    options, args = optparser.parse_args()

    left_date = datetime.strptime(options.left_date, '%Y-%m-%d').date()
    right_date = datetime.strptime(options.right_date, '%Y-%m-%d').date()

    create_current_file_run_log()

    if options.verbose:
        add_stdout_handler(log)
    else:
        yt_logger_config.LOG_LEVEL = 'ERROR'
        reload(yt_logger)

    configure_wrapper(yt)

    try:
        import_redirects(left_date, right_date, options.check_last_run)
    except Exception:
        log.exception('Unexpected error in import_redirects')
        raise


if __name__ == '__main__':
    main()
