# encoding: utf-8
import os
import logging
from datetime import datetime

import yt.wrapper as yt
import yt.yson as yson


logger = logging.getLogger(__name__)


YT_REVISE_PATH = '//home/rasp/revise/'

SCHEMA = yson.YsonList([
    {'name': 'unixtime', 'type': 'uint64'},
    {'name': 'partner', 'type': 'string'},
    {'name': 'result', 'type': 'string'},
    {'name': 'hit_time', 'type': 'string'},
    {'name': 'review_time', 'type': 'string'},
    {'name': 'point_from', 'type': 'string'},
    {'name': 'point_to', 'type': 'string'},
    {'name': 'date_forward', 'type': 'string'},
    {'name': 'date_backward', 'type': 'string'},
    {'name': 'klass', 'type': 'string'},
    {'name': 'adults', 'type': 'int64'},
    {'name': 'children', 'type': 'int64'},
    {'name': 'infants', 'type': 'int64'},
    {'name': 'national_version', 'type': 'string'},
    {'name': 'price', 'type': 'double'},
    {'name': 'currency', 'type': 'string'},
    {'name': 'price_revise', 'type': 'double'},
    {'name': 'currency_revise', 'type': 'string'},
    {'name': 'price_diff_abs', 'type': 'double'},
    {'name': 'price_diff_rel', 'type': 'double'},
    {'name': 'order_content', 'type': 'string'},
    {'name': 'revise_data', 'type': 'string'},
    {'name': 'search_depth', 'type': 'int64'},
    {'name': 'utm_source', 'type': 'string'},
    {'name': 'query_source', 'type': 'string'},
    # Эмулируем доставку из Логфеллера
    {'name': '_rest', 'type': 'any'},
    {'name': '_logfeller_index_bucket', 'type': 'string'},
    {'name': '_logfeller_timestamp', 'type': 'uint64'},
    {'name': '_stbx', 'type': 'string'},
    {'name': 'source_uri', 'type': 'string'},
])
SCHEMA.attributes['strict'] = False


def convert_db_record_to_yt_record(record):
    return {
        'unixtime': int((record.review_time.replace(tzinfo=None) - datetime(1970, 1, 1)).total_seconds()),
        'partner': record.partner,
        'result': record.result,
        'hit_time': record.hit_time.strftime('%Y-%m-%d %H:%M:%S'),
        'review_time': record.review_time.strftime('%Y-%m-%d %H:%M:%S'),
        'point_from': record.point_from,
        'point_to': record.point_to,
        'date_forward': record.date_forward.strftime('%Y-%m-%d'),
        'date_backward': record.date_backward.strftime('%Y-%m-%d') if record.date_backward else None,
        'klass': record.klass,
        'adults': record.adults,
        'children': record.children,
        'infants': record.infants,
        'national_version': record.national_version,
        'price': float(record.price),
        'currency': record.currency,
        'price_revise': record.price_revise,
        'currency_revise': record.currency_revise,
        'price_diff_abs': record.price_diff_abs,
        'price_diff_rel': record.price_diff_rel,
        'order_content': record.order_content,
        'revise_data': record.revise_data,
        'search_depth': record.search_depth,
        'utm_source': record.utm_source,
        'query_source': record.query_source,
        # Симулируем доставку из Логфеллера
        '_rest': None,
        '_logfeller_index_bucket': None,
        '_logfeller_timestamp': None,
        '_stbx': None,
        'source_uri': 'from_db',
    }


def write_group_to_yt(date, records):
    table_path = os.path.join(YT_REVISE_PATH, date.strftime('%Y-%m-%d'))
    if not yt.exists(table_path):
        yt.create_table(table_path, recursive=True, attributes={
            'schema': SCHEMA,
            'optimize_for': 'scan',
        })

    table_path = yt.TablePath(table_path, append=True)
    yt.write_table(table_path, records)


def main():
    import os

    os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'travel.avia.stat_admin.local_settings')
    import django
    django.setup()

    from travel.avia.stat_admin.lib.logs import add_stdout_handler
    import travel.avia.stat_admin.lib.yt_helpers as yth
    from travel.avia.stat_admin.data.models import PartnerReview

    yth.configure_wrapper(yt)

    add_stdout_handler(logger)
    logger.setLevel(logging.INFO)

    prev_date = None
    stacked_records = []
    total_records = 0

    logger.info('Start')
    for record in PartnerReview.objects.all().order_by('review_time'):
        date = record.review_time.date()

        if prev_date is not None and date != prev_date:
            logger.info('Write date [%s] to YT', prev_date)
            write_group_to_yt(prev_date, stacked_records)
            logger.info('%d rows written', len(stacked_records))
            total_records += len(stacked_records)

            stacked_records = [convert_db_record_to_yt_record(record)]

        else:
            stacked_records.append(convert_db_record_to_yt_record(record))

        prev_date = date

    logger.info('Write date [%s] to YT', prev_date)
    write_group_to_yt(prev_date, stacked_records)
    logger.info('%d rows written', len(stacked_records))
    total_records += len(stacked_records)

    logger.info('Total records: %d', total_records)


if __name__ == '__main__':
    main()
