# -*- encoding: utf-8 -*-
import os

from collections import defaultdict
from datetime import datetime
from optparse import OptionParser

import yt.wrapper as yt


AVIATICKETS_ROOT = '//home/rasp/logs/aurora_aviatickets-log'


def main():
    os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'travel.avia.stat_admin.local_settings')
    import django
    django.setup()

    import logging

    from django.db import transaction
    from django.db.models import Min

    from travel.avia.stat_admin.data.models import RivalPosition

    from travel.avia.stat_admin.lib.yt_helpers import configure_wrapper, yt_last_logs_tables

    @transaction.atomic()
    def update_rivals(eventdate, bulk_data):
        log.info('Clean rivals')
        RivalPosition.objects.filter(eventdate=eventdate).delete()

        log.info('Write rivals data to DB')
        RivalPosition.objects.bulk_create(bulk_data, batch_size=1000)

    def apply_positions(records):
        records = sorted(records, key=lambda x: x['minimal_price'])

        position = 0
        last_price = None

        for r in records:
            if not last_price or r['minimal_price'] != last_price:
                position += 1
                last_price = r['minimal_price']

            r['position'] = position

        return records

    log = logging.getLogger(__name__)

    optparser = OptionParser()

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-d', '--days', dest="days", type="int", default=2)

    options, args = optparser.parse_args()

    log.info('Start')
    configure_wrapper(yt)

    source_tables = yt_last_logs_tables(AVIATICKETS_ROOT, days=options.days)

    for source_table in source_tables:
        data = defaultdict(list)
        table_date = datetime.strptime(source_table.split('/')[-1], '%Y-%m-%d').date()
        for record in yt.read_table(source_table, format=yt.JsonFormat(attributes={'encode_utf8': False}), raw=False):
            direction = '{origin} - {destination}'.format(
                origin=record.get('origin').encode('utf8'),
                destination=record.get('destination').encode('utf8'),
            )

            passengers = '{adults}_{child}_{infant}'.format(
                adults=record.get('adults'),
                child=record.get('child'),
                infant=record.get('infant'),
            )

            one_way = record.get('one_way_ticket')
            one_way = True if one_way == 'true' else False

            record_key = (direction, passengers, one_way)

            record['minimal_price'] = int(float(record['minimal_price']))

            if record['minimal_price']:
                data[record_key].append(record)

        bulk_data = []
        for record_key, records in data.items():
            records = apply_positions(records)

            for position, record in enumerate(records):
                direction, passengers, one_way = record_key
                site_name = record.get('_source_table')
                partner = record.get('partner', '')
                position = record.get('position')
                minimal_price = record.get('minimal_price')
                avia_company = record.get('aviacompany', "").replace("&nbsp;", "")

                if minimal_price:
                    rp = RivalPosition(
                        eventdate=table_date,
                        rival_name=site_name,
                        partner=partner,
                        direction=direction,
                        passengers=passengers,
                        one_way=one_way,
                        position=position,
                        minimal_price=minimal_price,
                        avia_company=avia_company,
                    )

                    bulk_data.append(rp)

        update_rivals(table_date, bulk_data)

        log.info('Find and mark cheaper partners')
        rivals = RivalPosition.objects.filter(
            eventdate=table_date
        ).values('direction').annotate(Min('minimal_price')).order_by('direction')

        for r in rivals:
            cheaper_rivals = RivalPosition.objects.filter(
                eventdate=table_date,
                direction=r['direction'],
                minimal_price=r['minimal_price__min']
            )

            for cr in cheaper_rivals:
                cr.cheaper = True
                cr.save()

    log.info('Done')


if __name__ == '__main__':
    main()
