# -*- encoding: utf-8 -*-
import logging
import os
import sys
from datetime import datetime
from optparse import OptionParser

import yt.wrapper as yt


def reduce(key, records):
    count = 0
    query_time_sum = 0
    for r in records:
        count += 1
        query_time_sum = query_time_sum + float(r['query_time'])

    yield {
        'service': key['service'],
        'partner': key['partner'],
        'status': key['status'],
        'iso_eventtime': key['iso_eventtime'][:16],
        'count': count,
        'avg_query_time': int(query_time_sum) / count
    }


def main():
    os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'travel.avia.stat_admin.local_settings')
    import django
    django.setup()

    from django.db import transaction
    from travel.avia.stat_admin.data.models import PartnerQuery
    from django.utils import timezone

    from travel.avia.stat_admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.stat_admin.lib.yt_helpers import configure_wrapper, yt_last_logs_tables
    from travel.avia.stat_admin.utils.query import parse_qid

    log = logging.getLogger(__name__)

    def map(record):
        importer = record.get('importer', '')
        qid = record.get('qid', '')
        try:
            _when_created, service, _t_code, query_key, _lang = parse_qid(qid)
        except ValueError:
            sys.stderr.write('Can not parse qid {}\n'.format(qid))
            return

        status = record.get('status')
        iso_eventtime = '%s0' % record.get('iso_eventtime')[:15]

        if all([importer, qid, status, iso_eventtime]):
            try:
                partner_codes = importer.split('[')[1].strip(']').split(',')
            except IndexError:
                return

            variants_len = record.get('variants_len')
            query_time = record.get('query_time', 0)
            try:
                query_time = float(query_time)
            except ValueError:
                sys.stderr.write('Bad query_time: {}\n'.format(query_time))
                return

            if service:
                if any([pc.startswith('dohop_') for pc in partner_codes]):
                    partner_codes.append('dohop')

                for partner_code in partner_codes:
                    yield {
                        'service': service,
                        'partner': partner_code,
                        'status': status,
                        'iso_eventtime': iso_eventtime,
                        'query_time': query_time,
                    }

                # Add empty answers
                if not variants_len:
                    return

                try:
                    variants_len = int(variants_len)
                except ValueError:
                    sys.stderr.write('Bad variants len: {}\n'.format(variants_len))

                if status == 'got_reply' and variants_len == 0:
                    yield {
                        'service': service,
                        'partner': partner_code,
                        'status': 'got_empty_reply',
                        'iso_eventtime': iso_eventtime,
                        'query_time': query_time,
                    }

    log.info('Start')

    @transaction.atomic
    def save_results(tmp_table):
        for field in yt.read_table(tmp_table, format=yt.JsonFormat(), raw=False):
            current_tz = timezone.get_current_timezone()
            iso_eventtime = current_tz.localize(
                datetime.strptime(field['iso_eventtime'], '%Y-%m-%d %H:%M')
            )

            now = current_tz.localize(datetime.now())

            skip_current_hour = all([
                iso_eventtime.date() == now.date(),
                iso_eventtime.hour == now.hour,

            ])

            if not skip_current_hour:
                pq, created = PartnerQuery.objects.get_or_create(
                    query_datetime=iso_eventtime,
                    service=field['service'],
                    partner_code=field['partner'],
                    query_status=field['status'],
                    defaults={
                        'count': int(field['count']),
                        'avg_query_time': int(field['avg_query_time'])
                    }
                )

                if not created and int(pq.count) == int(field['count']):
                    continue

                pq.count = int(field['count'])
                pq.avg_query_time = int(field['avg_query_time'])
                pq.save()

    optparser = OptionParser()

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option("-d", "--days", dest="days", type="int", help="number of last logs to aggregate", default=2)

    options, args = optparser.parse_args()

    create_current_file_run_log()
    if options.verbose:
        add_stdout_handler(log)

    log.info('Start')
    configure_wrapper(yt)

    source_tables = yt_last_logs_tables('//home/rasp/logs/rasp-partners-query-log', options.days)

    tmp_table = yt.create_temp_table('//home/avia/tmp')
    log.info('Destination table: %s', tmp_table)

    yt.run_map_reduce(
        mapper=map,
        reducer=reduce,
        source_table=source_tables,
        destination_table=tmp_table,
        reduce_by=['service', 'partner', 'status', 'iso_eventtime'],
    )

    log.info('Saving results')
    save_results(tmp_table)

    log.info('Remove result table')
    yt.remove(tmp_table)

    log.info('End')


if __name__ == '__main__':
    main()
