# -*- encoding: utf-8 -*-
import os
from datetime import datetime
from optparse import OptionParser

import yt.wrapper as yt
import yt.logger as yt_logger


AVAILABLE_SERVICES = ['ticket', 'api']
RENAME_MAP = {'api_avia': 'api'}


def search_map(record):
    iso_eventtime = datetime.strptime(record['iso_eventtime'], '%Y-%m-%d %H:%M:%S')
    service = RENAME_MAP.get(record.get('host'), record.get('host'))

    if service in AVAILABLE_SERVICES:
        yield {
            'iso_eventtime': iso_eventtime.strftime('%Y-%m-%d %H:00:00'),
            'service': service,
            'pp': record.get('pp', ''),
            'national_version': record.get('national'),
            'count': 1
        }


def search_reduce(key, records):
    yield {
        'iso_eventtime': key['iso_eventtime'],
        'service': key['service'],
        'national_version': key['national_version'],
        'pp': key['pp'],
        'count': sum([int(r['count']) for r in records])
    }


def main():
    os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'travel.avia.stat_admin.local_settings')
    import django
    django.setup()

    import logging

    from django.conf import settings
    from django.utils import timezone

    from travel.avia.stat_admin.data.models import UserRedirect

    from travel.avia.stat_admin.lib.logs import add_stdout_handler
    from travel.avia.stat_admin.lib.yt_helpers import configure_wrapper, yt_last_logs_tables

    log = logging.getLogger(__name__)

    optparser = OptionParser()

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option("-d", "--days", dest="days", type="int", help="number of last logs to aggregate", default=2)

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)
    else:
        yt_logger.LOGGER.setLevel(logging.ERROR)

    log.info('Start')
    configure_wrapper(yt)
    source_tables = yt_last_logs_tables('//home/avia/logs/avia-json-redir-log', options.days)

    tmp_table = yt.create_temp_table(
        path=settings.YT_TMP_DIRECTORY,
        prefix='avia_stats_'
    )

    yt.run_map_reduce(
        mapper=search_map,
        reducer=search_reduce,
        source_table=source_tables,
        destination_table=tmp_table,
        reduce_by=['iso_eventtime', 'service', 'national_version', 'pp'],
    )

    for field in yt.read_table(tmp_table, format=yt.JsonFormat(), raw=False):
        current_tz = timezone.get_current_timezone()
        iso_eventtime = current_tz.localize(
            datetime.strptime(field['iso_eventtime'], '%Y-%m-%d %H:%M:%S')
        )

        now = current_tz.localize(datetime.now())

        skip_current_hour = all([
            iso_eventtime.date() == now.date(),
            iso_eventtime.hour == now.hour,

        ])

        if not skip_current_hour:
            new_search, created = UserRedirect.objects.get_or_create(
                eventdate=iso_eventtime,
                service=field['service'],
                partner_code='',
                national_version=field['national_version'],
                pp=field['pp'],
                defaults={
                    'count': int(field['count'])
                }
            )

            if not created:
                new_search.count = int(field['count'])

            new_search.save()

    yt.remove(tmp_table)

if __name__ == '__main__':
    main()
