# -*- encoding: utf-8 -*-
import argparse
import os
from datetime import datetime

import yt.wrapper as yt


def search_map(record):
    if 'yandexuid' in record and record['yandexuid']:
        iso_eventtime = datetime.strptime(record['iso_eventtime'], '%Y-%m-%d %H:%M:%S')
        yield {
            'iso_eventtime': iso_eventtime.strftime('%Y-%m-%d %H:00:00'),
            'national_version': record.get('national_version', 'ru'),
            'user_geoid': record.get('user_geoid'),
            'count': 1
        }


def search_reduce(key, records):
    yield {
        'iso_eventtime': key['iso_eventtime'],
        'national_version': key['national_version'],
        'user_geoid': key.get('user_geoid'),
        'count': sum([int(r['count']) for r in records])
    }


def main():
    os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'travel.avia.stat_admin.local_settings')
    import django
    django.setup()

    from django.conf import settings
    from django.utils import timezone

    import logging
    import geobase6 as geobase_lib

    from travel.avia.stat_admin.data.models import RedirectHeatMap
    from travel.avia.stat_admin.lib.yt_helpers import configure_wrapper, yt_last_logs_tables

    from travel.avia.stat_admin.lib.logs import create_current_file_run_log, add_stdout_handler

    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true')
    parser.add_argument('-d', '--days', default=2, type=int)
    args = parser.parse_args()

    create_current_file_run_log()

    log = logging.getLogger(__name__)

    if args.verbose:
        add_stdout_handler(log)

    log.info('Start')

    configure_wrapper(yt)

    logs_quantity = args.days

    source_tables = yt_last_logs_tables('//home/rasp/logs/rasp-popular-flights-log', logs_quantity)

    tmp_table = yt.create_temp_table(
        path=settings.YT_TMP_DIRECTORY,
        prefix='avia_stats_'
    )

    yt.run_map_reduce(
        mapper=search_map,
        reducer=search_reduce,
        source_table=source_tables,
        destination_table=tmp_table,
        reduce_by=['iso_eventtime', 'national_version', 'user_geoid'],
    )

    lookup = geobase_lib.Lookup("/var/cache/geobase/geodata6.bin")
    for field in yt.read_table(tmp_table, format=yt.JsonFormat(), raw=False):
        current_tz = timezone.get_current_timezone()
        iso_eventtime = current_tz.localize(
            datetime.strptime(field['iso_eventtime'], '%Y-%m-%d %H:%M:%S')
        )

        now = current_tz.localize(datetime.now())

        skip_current_hour = all([
            iso_eventtime.date() == now.date(),
            iso_eventtime.hour == now.hour,

        ])

        if not skip_current_hour:
            geoid = int(field["user_geoid"])
            region = lookup.region_by_id(geoid)

            new_search, created = RedirectHeatMap.objects.get_or_create(
                eventdate=iso_eventtime,
                national_version=field['national_version'],
                geo_id=geoid,
                defaults={
                    'count': int(field['count']),
                    'lat': region.lat,
                    'lon': region.lon,
                }
            )

            if not created:
                new_search.count = int(field['count'])

            new_search.save()

    yt.remove(tmp_table)

    log.info('Done')

if __name__ == '__main__':
    main()
