# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, unicode_literals

import bisect
import csv
import decimal
import json
import logging
import math
import os
import tempfile
import time
from datetime import datetime, timedelta
from itertools import count, islice, takewhile

import requests
from django.conf import settings
from django.core.management.base import BaseCommand
from django.template.loader import render_to_string
from django.utils.html import escape

from travel.avia.stat_admin.data.models.partner import PartnerEmail
from travel.avia.stat_admin.lib.mail import SmailikClient

from ._loggers import setup_logger_verbosity
from ._yt_send_partners_stat import calculate_stat_by_partners

log = logging.getLogger('scripts')


def _gen_pages_from_paginated_rest(url, params, timeout=1):
    page = requests.get(url, params=params, verify=False, timeout=timeout).json()
    yield page
    next_url = page.get('next')
    while next_url:
        page = requests.get(next_url, verify=False, timeout=timeout).json()
        yield page
        next_url = page.get('next')


def gen_items_from_paginated_rest(url, params):
    for page in _gen_pages_from_paginated_rest(url, params):
        for item in page['results']:
            yield item


def get_partners_names():
    if not settings.RASP_REST_HOST:
        log.info('No settings.RASP_REST_HOST')
        return {}
    return {
        p['code']: p['title']
        for p in gen_items_from_paginated_rest(
            settings.RASP_REST_HOST + '/rest/dyn/order/partner/',
            params={'fields': 'code,title'},
        )
    }


class Command(BaseCommand):
    help = 'Send to partners statistic'

    def handle(self, *args, **options):
        setup_logger_verbosity(log, options['verbosity'], self.stdout, self.stderr)
        log.debug('Begin')

        try:
            partners_names = get_partners_names()
        except Exception:
            log.error('Error in get_partners_names')
            partners_names = {}
        log.debug('partners_names: %s', partners_names)

        today = datetime.now().date()
        begin = today - timedelta(days=7)
        end = today - timedelta(days=1)

        qids_count, data = calculate_stat_by_partners(begin, end)

        pd_stat_by_partner = {
            p: pd_dataframe_from_list_of_dicts(stat)
            for p, stat in data.items()
        }

        week = {
            p: aggregate(partner_data)
            for p, partner_data in pd_stat_by_partner.items()
            if partner_data
        }
        week = {p: agg for p, agg in week.items() if agg['good']}

        for rate, p in enumerate(sorted(week.keys(), key=lambda p: week[p]['avg_query_time']), start=1):
            week[p]['avg_query_time_rate'] = rate, len(week)

        for rate, p in enumerate(sorted(week.keys(), key=lambda p: week[p]['error']), start=1):
            week[p]['error_rate'] = rate, len(week)

        for rate, p in enumerate(sorted(week.keys(), key=lambda p: week[p]['empty']), start=1):
            week[p]['empty_rate'] = rate, len(week)

        for rate, p in enumerate(sorted(week.keys(), key=lambda p: week[p]['timeout']), start=1):
            week[p]['timeout_rate'] = rate, len(week)

        # for rate, p in enumerate(sorted(week.keys(), key=lambda p: week[p]['redir_fails']), start=1):
        #     week[p]['redir_fails_rate'] = rate, len(week)

        def prepare_stat_for_csv(frame):
            frame['avg_query_time'] = [round_significant(val, 3) for val in frame['avg_query_time']]
            return frame

        yandex_search_count = qids_count
        partners_count = len(week)

        for pcode, pweek in week.items():
            try:
                send_partner_stat(
                    pcode,
                    partners_names.get(pcode) or pcode.capitalize(),
                    begin, end,
                    pweek,
                    prepare_stat_for_csv(pd_stat_by_partner[pcode]),
                    {
                        'yandex_search_count': yandex_search_count,
                        'partners_count': partners_count,
                    }
                )
            except Exception:
                log.exception('Error while sending stat to %s', pcode)

        log.info('Done')


def aggregate(stat):
    qs = sum(stat['queries'])
    good = sum(stat['good'])

    d = {
        'queries': qs,
        'good': good,
        'timeout': sum(stat['timeout']) * 100 / qs,
        'empty': sum(stat['empty']) * 100 / qs,
        'error': sum(stat['error']) * 100 / qs,
        # 'redir_fails': sum(stat['redir_fails']) * 100 / sum(stat['redirs']),
        'avg_query_time': good and summ(
            aqt * g
            for aqt, g in zip(stat['avg_query_time'], stat['good'])
            if g
        ) / good
    }
    return d


def summ(items):
    if not items:
        return 0
    return sum(items)


def round_significant(value, count=1):
    v = value or 0
    v = decimal.Decimal(v)
    try:
        if v <= 0:
            return v
    except decimal.InvalidOperation:
        return value
    return round(v, count - 1 - int(math.floor(math.log10(v))))


def pd_dataframe_from_list_of_dicts(data):
    columns = data[0].keys()
    series = {k: [row.get(k) for row in data] for k in columns}
    return DataFrame(series, columns=columns)


class DataFrame(dict):
    def __init__(self, series, columns):
        super(DataFrame, self).__init__(series)
        self.columns = columns

    def to_csv(self, filepath, datetime_format, index=False):
        with open(filepath, 'wb') as f:
            writer = DictWriterTypesAware(
                f,
                fieldnames=self.columns,
                encoding='utf8',
                delimiter=','.encode('utf-8'),
                datetime_format=datetime_format
            )
            writer.writeheader()
            for row in zip(*[self[field] for field in self.columns]):
                writer.writerow(dict(zip(self.columns, row)))


class DictWriterTypesAware(csv.DictWriter):
    def __init__(self, *args, **kwargs):
        self.encoding = kwargs.pop('encoding', 'utf8')
        self.errors = kwargs.pop('errors', 'strict')
        self.datetime_format = kwargs.pop('datetime_format', '%Y-%m-%d %H:%M:%S')
        csv.DictWriter.__init__(self, *args, **kwargs)

    def _dict_to_list(self, rowdict):
        for k, value in rowdict.iteritems():
            rowdict[k] = self.convert_value(value)
        return csv.DictWriter._dict_to_list(self, rowdict)

    def convert_value(self, value):
        if isinstance(value, datetime):
            value = value.strftime(self.datetime_format)
        if isinstance(value, unicode):
            return value.encode(self.encoding, self.errors)
        elif value is None:
            return ""
        else:
            return unicode(value).encode(self.encoding, self.errors)


GREEN = '#28a528'
GREY = '#666666'
RED = '#c82828'


Z = .1
for j in range(30):
    Z = 1/(1+Z)
Z4 = Z**4


RATES_POS, COLORS = zip(*sorted({
    1: GREEN,
    1-Z4: GREY,
    Z4: RED,
}.items()))


def rate_color(pos):
    num, den = pos
    return COLORS[bisect.bisect(RATES_POS, 1-num/den)]


def send_partner_stat(pcode, pname, begin, end, week, stat, ctx):
    log.debug('send_partner_stat: %s [%s - %s] %s', pcode, begin, end, week)

    emailtype = 'stat'

    emails = [
        pe.email
        for pe in PartnerEmail.objects.filter(
            partner_code__code=pcode, emailtype__code=emailtype)
    ]
    if not emails:
        log.warning('Email [%s] not found for partner: %s', emailtype, pcode)
        return

    ctx['now'] = datetime.now()
    daterange_human = format_daterange(begin, end)
    subject = 'Статистика по работе вашего API за {}'.format(daterange_human)

    ctx['daterange_human'] = daterange_human

    ctx['big_table'] = [
        ('хороших ответов', '{}%'.format(round_significant(week['good'] * 100 / week['queries'], count=3))),
        ('запросов к сервису', format_big(round_significant(week['queries'], count=3))),
        ('ответов от сервиса', format_big(round_significant(week['good'], count=3))),
    ]

    ctx['table'] = [
        # title, color, value, rate
        (
            'Среднее время ответа',
            '{}с'.format(format_big(round_significant(week['avg_query_time'], count=3))) if week['avg_query_time'] else '—',
            week['avg_query_time_rate'],
            rate_color(week['avg_query_time_rate'])
        ),
        (
            'Не нашлось варианта',
            '{}%'.format(round_significant(week['empty'], count=2)),
            week['empty_rate'],
            rate_color(week['empty_rate'])
        ),
        (
            'Сервис не ответил',
            '{}%'.format(round_significant(week['timeout'], count=2)),
            week['timeout_rate'],
            rate_color(week['timeout_rate'])
        ),
        (
            'Ошибка в формате данных',
            '{}%'.format(round_significant(week['error'], count=2)),
            week['error_rate'],
            rate_color(week['error_rate'])
        ),
        # (
        #     'Ошибка при переадресации',
        #     RED,
        #     round_significant(week['redir_fails'], count=2),
        #     week['redir_fails_rate'], rate_color(week['redir_fails_rate'])
        # ),
    ]

    mail_content = render_to_string('partner_stat.xml', ctx)
    csv_name = 'yandex_flights_{}_{}.csv'.format(
        begin.strftime('%Y-%m-%d'), end.strftime('%Y-%m-%d')
    )

    for email in emails:
        send_partner_mail(pcode, pname, email, subject, mail_content, stat, csv_name)
        time.sleep(5)


MONTH = [''] + u"""
января февраля марта апреля мая июня июля августа сентября октября ноября декабря
""".split()


def format_daterange(begin, end):
    begin, end = sorted([begin, end])
    yb, mb, db = begin.timetuple()[:3]
    ye, me, de = end.timetuple()[:3]

    y = (yb != ye) and yb
    m = (y or mb != me) and mb
    d = (m or db != de) and db

    fbegin = ' '.join(filter(None, [
        d and str(d),
        m and MONTH[m],
        y and '{} года'.format(y)
    ]))

    fend = ' '.join(filter(None, [
        str(de),
        MONTH[me],
        ye and '{} года'.format(ye)
    ]))

    return ' — '.join(filter(None, [fbegin, fend]))


def test_format_daterange():
    assert format_daterange(
        datetime(2016, 4, 28),
        datetime(2016, 5, 5)
    ) == '28 апреля — 5 мая 2016 года'

    assert format_daterange(
        datetime(2016, 5, 5),
        datetime(2015, 4, 28)
    ) == '28 апреля 2015 года — 5 мая 2016 года'

    assert format_daterange(
        datetime(2016, 5, 5),
        datetime(2016, 5, 7)
    ) == '5 — 7 мая 2016 года'


def format_big(val):
    val *= 1000
    for one in ['', ' тыс', ' млн', ' млрд']:
        val /= 1000
        if val < 1000:
            return '{}{}'.format(val, one)
    return '{}{}'.format(format_int(int(val)), one)


def test_format_big_int():
    assert format_big(34) == '34.0'
    assert format_big(34 * 10**3) == '34.0 тыс'
    assert format_big(34 * 10**6) == '34.0 млн'
    assert format_big(34 * 10**9) == '34.0 млрд'
    assert format_big(34111 * 10**9) == '34 111 млрд'


def format_int(val):
    j = iter(reversed('%s' % int(val)))
    return (' '.join(takewhile(len, (''.join(islice(j, 3)) for _ in count()))))[::-1]


def send_partner_mail(pcode, pname, email, subject, content, stat, csv_name):
    c = SmailikClient(
        service='Flights.Yandex',
        send_from='Яндекс.Авиабилеты &lt;avia-partner@support.yandex.ru&gt;',
        reply_to='Яндекс.Авиабилеты &lt;avia-info@yandex-team.ru&gt;',
        smailik_dir='/var/spool/smailik-preparer/avia-stats/'
    )

    d = tempfile.mkdtemp(prefix='avia-stat-attachments-')
    filepath = os.path.join(d, csv_name)
    log.debug('csv file: %s', filepath)
    stat.to_csv(filepath, index=False, datetime_format='%Y-%m-%d %H:%M:%S')

    email = '{} &lt;{}&gt;'.format(escape(pname), email)
    c.send_email(
        email,
        escape(subject),
        content,
        [('text/csv', filepath)],
        bcc=settings.SEND_PARTNER_STAT_EMAIL_BCC
    )

    with open(filepath) as f:
        attachment = f.read()

    logging.getLogger('sent_email').info(json.dumps([
        {
            'email': email,
            'subject': subject,
            'bcc': settings.SEND_PARTNER_STAT_EMAIL_BCC,
        }, {
            'content': content,
        }, {
            'attachments': [attachment],
        }
    ]))

    log.info('Email sent for %s to %s', pcode, email)
