# -*- encoding: utf-8 -*-
import csv
import gzip
import functools
import dateutil.parser
import logging
import io
import os
import requests
import sys
from copy import copy
from datetime import datetime
from optparse import Option, OptionParser, OptionValueError

import yt.wrapper as yt


def logs_tables(path, from_date, to_date):
    tables = []
    for table in yt.search(path, node_type="table"):
        table_date = datetime.strptime(table.split('/')[-1], '%Y-%m-%d').date()

        if from_date <= table_date <= to_date:
            tables.append(table)

    return tables


def fast_datetime_convert(pydate):
    """
    20131203010651
    year, month, day, hour, minute, second
    """
    return datetime(
        int(pydate[0:4]), int(pydate[4:6]), int(pydate[6:8]),
        int(pydate[8:10]), int(pydate[10:12]), int(pydate[12:14])
    )


def parse_line(text):
    return dict([var.split('=') for var in text.strip().split('@@')])


def check_number(option, opt, value):
    try:
        days=int(value)

    except ValueError:
        raise OptionValueError(
            "option %s: bad int: %r" % (opt, value))

    return days


def check_date(option, opt, value):
    try:
        date_time=dateutil.parser.parse(value)

    except ValueError:
        raise OptionValueError(
            "option %s: bad date: %r" % (opt, value))

    return date_time.date()


def fix_point(point_key, stations_map):
    if point_key.startswith('s'):
        return stations_map.get(point_key)

    return point_key


def check_options(options):
    if not options.start_date:
        print u'Задайте начальную дату: -s'
        sys.exit(1)

    if not options.end_date:
        print u'Задайте конечную дату: -e'
        sys.exit(1)

    if options.start_date > options.end_date:
        print u'Конечная дата должна быть больше начальной'
        sys.exit(1)

    if not options.mailto:
        print u'Задайте получателя: -m'
        sys.exit(1)


def main():
    def map_tables(partner, record):
        if partner:
            if partner == 'dohop' and record.get('partner', '').startswith('dohop'):
                yield record

            elif partner != 'dohop' and record.get('partner') == partner:
                yield record

        else:
            yield record

    def make_and_send_results(tmp_table, options, point_keys_to_iata_map, storage_dir):
        file_name = os.path.join(
            storage_dir,
            'redirects%s.csv' % range_key
        )
        csvfile = file(file_name, 'w')

        csv_writer = csv.writer(csvfile, delimiter=';', quotechar='"', quoting=csv.QUOTE_ALL)
        header = ['Partner', 'Eventtime', 'Departure', 'Arrival', 'Date forward', 'Date backward', 'Forward flights', 'Backward flights', 'National version', 'Price', 'Currency']
        csv_writer.writerow(header)

        for record in yt.read_table(tmp_table, format=yt.DsvFormat(), raw=False):
            try:
                price, currency = record.get('tariff').split(' ')
            except (ValueError, AttributeError):
                log.info("Bad tariff value '%r'", record.get('tariff'))
                continue

            row = [
                record.get('partner'),
                record.get('iso_eventtime'),
                point_keys_to_iata_map.get(record.get('from_id'), record.get('from_id')),
                point_keys_to_iata_map.get(record.get('to_id'), record.get('to_id')),
                record.get('date_forward'),
                record.get('date_backward'),
                record.get('forward_numbers'),
                record.get('backward_numbers'),
                record.get('national_version'),
                int(float(price)),
                currency
            ]
            csv_writer.writerow(row)

        csvfile.flush()
        csvfile.close()

        mail = EmailMultiAlternatives(
            subject=u'Переходы %s' % (range_key.replace('_', ' - ')),
            body=u'Даты: %s\n\nФайлы во вложении:\n\n' % (range_key.replace('_', ' - ')),
            from_email=settings.SERVER_EMAIL,
            to=['%s@yandex-team.ru' % options.mailto],
        )

        with open(csvfile.name, 'rb') as f, io.BytesIO() as b:
            g = gzip.GzipFile(mode='wb', fileobj=b, compresslevel=9)
            g.writelines(f)
            g.close()
            attachment = MIMEApplication(b.getvalue(), 'x-gzip')
            attachment['Content-Disposition'] = 'attachment; filename=%s.gz' % os.path.basename(csvfile.name)

        mail.attach(attachment)

        try:
            mail.send()

        except Exception:
            log.exception("ERROR")

    # START MAIN
    from email.mime.application import MIMEApplication

    os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'travel.avia.stat_admin.local_settings')
    import django
    django.setup()

    from django.conf import settings
    from django.core.mail.message import EmailMultiAlternatives

    from travel.avia.stat_admin.lib.jobs import get_or_create_job_storage_dir
    from travel.avia.stat_admin.lib.yt_helpers import configure_wrapper

    optparser = OptionParser(option_class=Yoption)

    log = logging.getLogger(__name__)

    optparser.add_option("-s", "--start_date", dest="start_date", type="date", help="date from")
    optparser.add_option("-e", "--end_date", dest="end_date", type="date", help="date to", default=datetime.now().date())
    optparser.add_option("-m", "--mailto", dest="mailto", type="email", help="Send resultts to")
    optparser.add_option("-p", "--partner", dest="partner", help="partner")
    optparser.add_option("-j", "--job", dest="job", type="number", help="job ID")

    options, args = optparser.parse_args()

    check_options(options)

    log.info('Start')

    range_key = '%s_%s' % (options.start_date, options.end_date)

    log.info('Dates: %s' % (range_key.replace('_', ' - ')))
    log.info('Email: %s' % options.mailto)

    configure_wrapper(yt)

    log.info('Check output dir')
    storage_dir = get_or_create_job_storage_dir(options.job)

    log.info('Prepare data')

    settlements = requests.get('%s/settlement/list/' % settings.RASP_API_HOST, verify=False).json()
    airports = requests.get('%s/station/airports/' % settings.RASP_API_HOST, verify=False).json()
    s2s = requests.get('%s/station2settlement/dict/' % settings.RASP_API_HOST, verify=False).json()

    point_keys_to_iata_map = {}

    for k, a in airports.items():
        if a['iata']:
            point_keys_to_iata_map[k] = a['iata']
            if a['settlement_point_key']:
                point_keys_to_iata_map[a['settlement_point_key']] = a['iata']

    for k, s in settlements.items():
        if s['iata']:
            point_keys_to_iata_map[k] = s['iata']

    for k, s in s2s.items():
        iata = point_keys_to_iata_map.get(s)
        if iata:
            point_keys_to_iata_map[k] = iata

    range_key = '%s_%s' % (options.start_date, options.end_date)

    yt_tables = logs_tables(
        '//home/rasp/logs/rasp-popular-flights-log',
        options.start_date,
        options.end_date,
    )

    tmp_table = yt.create_temp_table()

    log.info('Map tables')

    yt.run_map(
        functools.partial(map_tables, options.partner),
        source_table=yt_tables,
        destination_table=tmp_table,
    )

    yt.run_merge(
        source_table=tmp_table,
        destination_table=tmp_table,
    )

    log.info('Send results to email')
    try:
        make_and_send_results(tmp_table, options, point_keys_to_iata_map, storage_dir)

    except Exception:
        log.exception("ERROR")

    log.info('Done')

    yt.remove(tmp_table)


class Yoption(Option):
    TYPES = Option.TYPES + ("date", "number", "email")
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
    TYPE_CHECKER["date"] = check_date
    TYPE_CHECKER["number"] = check_number


if __name__ == '__main__':

    main()
