# -*- coding: utf-8 -*-
import codecs
import logging
import os.path
import sys
import traceback
from datetime import datetime

# !!!! Logging hack
logging._srcfile = os.path.normcase(logging.log.func_code.co_filename)


class FileProxyMixin(object):
    """
    Copypaste from django.core.files.utils

    A mixin class used to forward file methods to an underlaying file
    object.  The internal file object has to be called "file"::

        class FileProxy(FileProxyMixin):
            def __init__(self, file):
                self.file = file
    """

    encoding = property(lambda self: self.file.encoding)
    fileno = property(lambda self: self.file.fileno)
    flush = property(lambda self: self.file.flush)
    isatty = property(lambda self: self.file.isatty)
    newlines = property(lambda self: self.file.newlines)
    read = property(lambda self: self.file.read)
    readinto = property(lambda self: self.file.readinto)
    readline = property(lambda self: self.file.readline)
    readlines = property(lambda self: self.file.readlines)
    seek = property(lambda self: self.file.seek)
    softspace = property(lambda self: self.file.softspace)
    tell = property(lambda self: self.file.tell)
    truncate = property(lambda self: self.file.truncate)
    write = property(lambda self: self.file.write)
    writelines = property(lambda self: self.file.writelines)
    xreadlines = property(lambda self: self.file.xreadlines)

    def __iter__(self):
        return iter(self.file)


def patchedHandleError(self, record):
    """
    Handle errors which occur during an emit() call.

    This method should be called from handlers when an exception is
    encountered during an emit() call. If raiseExceptions is false,
    exceptions get silently ignored. This is what is mostly wanted
    for a logging system - most users will not care about errors in
    the logging system, they are more interested in application errors.
    You could, however, replace this with a custom handler if you wish.
    The record which was being processed is passed in to this method.
    """
    if logging.raiseExceptions and sys.stderr:  # see issue 13807
        ei = sys.exc_info()
        try:
            traceback.print_exception(ei[0], ei[1], ei[2],
                                      None, sys.stderr)
            traceback.print_stack(None, None, sys.stderr)
            sys.stderr.write('%s Logged from file %s, line %s\n' % (
                             datetime.now(), record.filename, record.lineno))
        except IOError:
            pass    # see issue 5971
        finally:
            del ei

logging.Handler.handleError = patchedHandleError


class WatchedFileHandler(logging.FileHandler):
    def _open(self):
        self.encoding = self.encoding or 'utf-8'

        return WatchedFile(self.baseFilename, mode=self.mode, encoding=self.encoding)


class WatchedFile(FileProxyMixin):
    """Файл, которые переоткрывается при его переименовании или удалении"""

    def __init__(self, filename, mode='a', encoding='utf-8'):
        self.file = None
        self.mode = mode
        self._encoding = encoding

        self.filename = filename
        self.dirname = os.path.dirname(self.filename)

        if not os.path.exists(self.filename):
            self.dev, self.ino = None, None
        else:
            stat = os.stat(self.filename)
            self.dev, self.ino = stat.st_dev, stat.st_ino

        self.open()

    def open(self):
        if not os.path.exists(self.dirname):
            os.makedirs(self.dirname)

        self.close()

        self.file = codecs.open(self.filename, self.mode, encoding=self._encoding)

        st = os.stat(self.filename)
        self.dev, self.ino = st.st_dev, st.st_ino

    def reopen(self):
        self.open()

    def flush(self):
        self.file and self.file.flush()

    def write(self, msg):
        if not os.path.exists(self.filename):
            changed = True

        else:
            st = os.stat(self.filename)
            the_same = (st.st_dev == self.dev) and (st.st_ino == self.ino)

            changed = not the_same

        if changed:
            self.reopen()

        if isinstance(msg, unicode):
            self.file.write(msg)
        else:
            self.file.write(unicode(msg, 'utf-8', 'replace'))

    def close(self):
        if not self.file:
            return

        if not self.file.closed:
            self.file.flush()
            self.file.close()


class FileTreeHandler(logging.Handler):
    """
    Обработчик, создающий дерево из обрабатываемых логов
    """

    SUFFIX = '.log'

    def __init__(self, base_path):
        """
        Initialize the handler.
        """
        logging.Handler.__init__(self)
        self.base_path = base_path
        self.streams = {}

    def get_stream(self, name):
        if name in self.streams:
            return self.streams[name]

        parts = name.split('.')

        path = os.path.join(self.base_path, *parts) + self.SUFFIX

        stream = self.streams[name] = WatchedFile(path)

        return stream

    def flush(self):
        """
        Flushes the streams.
        """

        for stream in self.streams.values():
            stream.flush()

    def emit(self, record):
        """
        Emit a record.

        If a formatter is specified, it is used to format the record.
        The record is then written to the stream with a trailing newline.
        """

        try:
            msg = self.format(record)

            stream = self.get_stream(record.name)

            stream.write("%s\n" % msg)

            stream.flush()

        except (KeyboardInterrupt, SystemExit):
            raise
        except:
            self.handleError(record)


class ArcadiaFileTreeHandler(FileTreeHandler):
    """
    Обработчик, создающий дерево из обрабатываемых логов
    с возможностью обрезать ненужную часть пути до файлов,
    например 'travel.avia.backend'
    """

    def __init__(self, base_path, remove_prefix=None):
        super(ArcadiaFileTreeHandler, self).__init__(base_path)
        self.remove_prefix = remove_prefix
        self.base_path = base_path
        self.streams = {}

    def get_stream(self, name):
        if name in self.streams:
            return self.streams[name]

        if self.remove_prefix and name.startswith(self.remove_prefix):
            cut_length = len(self.remove_prefix) if self.remove_prefix.endswith('.') else len(self.remove_prefix) + 1
            parts = name[cut_length:].split('.')
        else:
            parts = name.split('.')

        path = os.path.join(self.base_path, *parts) + self.SUFFIX

        stream = self.streams[name] = WatchedFile(path)

        return stream
