# -*- coding: utf-8 -*-

import logging

import MySQLdb
from django.utils.functional import cached_property
from django.db.backends.mysql.base import DatabaseWrapper as MysqlWrapper
from django.db.utils import DatabaseErrorWrapper as DjangoDBErrorWrapper, OperationalError
from travel.library.python.avia_mdb_replica_info.avia_mdb_replica_info import containers


log = logging.getLogger(__name__)


class DatabaseWrapper(MysqlWrapper):
    """Mysql коннектор. Автоматически ищет доступного мастера по списку реплик"""

    @cached_property
    def wrap_database_errors(self):
        """
        Context manager and decorator that re-throws backend-specific database
        exceptions using Django's common wrappers.
        """
        return DatabaseErrorWrapper(self)

    def define_host(self):
        conn_params = self.get_connection_params()
        actualize_replicas_info(
            replicas=self.settings_dict['REPLICAS'],
            kwargs=conn_params,
        )
        for replica in self.settings_dict['REPLICAS']:
            if replica.is_master:
                log.info('Redefine database host to %s', replica.hostname)
                self.settings_dict['HOST'] = replica.hostname
                break


def actualize_replicas_info(replicas, kwargs):
    log.info('Actualize replica information')
    for replica in replicas:  # type: containers.Replica
        try:
            actualize_replica_is_master(
                replica,
                specified_replica_kwargs(replica, kwargs)
            )
            log.info('%r is master: %s', replica, replica.is_master)
        except MySQLdb.OperationalError, e:
            log.warning('Could not connect to %s: %s', replica, e)


def actualize_replica_is_master(replica, kwargs):
    with MySQLdb.connect(**kwargs) as cursor:
        cursor.execute('show slave status')
        replica.is_master = not cursor.fetchall()


def specified_replica_kwargs(replica, kwargs):
    kwargs = kwargs.copy()
    kwargs['host'] = replica.hostname
    return kwargs


class DatabaseErrorWrapper(DjangoDBErrorWrapper):
    def __exit__(self, exc_type, exc_value, traceback):
        try:
            super(DatabaseErrorWrapper, self).__exit__(exc_type, exc_value, traceback)
        except OperationalError, e:
            if is_read_only_error(e):
                redefine_host_if_needed(self.wrapper)
            raise


def is_read_only_error(exc):
    message = 'The MySQL server is running with the --read-only option so it cannot execute this statement'
    return isinstance(exc, OperationalError) and exc.args[1] == message


def redefine_host_if_needed(db_wrapper):
    # type: (DatabaseWrapper) -> None
    db_wrapper.close()
    db_wrapper.define_host()
