import logging

import travel.avia.subscriptions.app.model.db as db_models
from travel.avia.subscriptions.app.api.init_db import Database
from travel.avia.subscriptions.app.api.interactor.user_confirm import UserConfirmActor
from travel.avia.subscriptions.app.api.interactor.user_price_change_subscription import UserPriceChangeSubscriptionActor
from travel.avia.subscriptions.app.api.interactor.user_promo_subscription import UserPromoSubscriptionActor
from travel.avia.subscriptions.app.api.interactor.user_subscription import UserSubscriptionActor
from travel.avia.subscriptions.app.api.interactor.user_subscription_list import UserSubscriptionListActor
from travel.avia.subscriptions.app.api.interactor.user_token import UserTokenActor
from travel.avia.subscriptions.app.lib.dicts import PointKeyResolver
from travel.avia.subscriptions.app.lib.pingtools import PingResponse
from travel.avia.subscriptions.app.lib.sender import TransactionalApi as Sender
from travel.avia.subscriptions.app.model.storage import DatabaseStorage
from travel.avia.subscriptions.app.settings.app import AppConfig

from blackbox import XmlBlackbox

logger = logging.getLogger(__name__)


class App:
    def __init__(self, config: AppConfig):
        self.config: AppConfig = config
        self.database: Database = Database(config, echo=config.debug)

        self.point_key_resolver = PointKeyResolver(oauth_token=config.sandbox_token)
        self.blackbox = XmlBlackbox(
            url=config.blackbox_url,
            tvm2_client_id=config.tvm_client_id,
            blackbox_client=config.blackbox_client,
        )
        self.user_token_actor: UserTokenActor = UserTokenActor(
            email=DatabaseStorage(db_models.Email),
            user_auth_type=DatabaseStorage(db_models.UserAuthType),
            user_auth=DatabaseStorage(db_models.UserAuth),
            user_promo_subscription=DatabaseStorage(db_models.UserPromoSubscription),
            user=DatabaseStorage(db_models.User),
            session_provider=self.database.session,
        )
        self.user_confirm_actor: UserConfirmActor = UserConfirmActor(
            user=DatabaseStorage(db_models.User),
            email=DatabaseStorage(db_models.Email),
            user_token_actor=self.user_token_actor,
            single_opt_in_sender=Sender(self.config.sender, self.config.sender.single_opt_in_campaign_slug),
            double_opt_in_sender=Sender(self.config.sender, self.config.sender.double_opt_in_campaign_slug),
            session_provider=self.database.session,
            blackbox=self.blackbox,
            force_confirm=self.config.business.force_user_confirm,
        )
        self.user_promo_subscription_actor: UserPromoSubscriptionActor = UserPromoSubscriptionActor(
            promo_subscription=DatabaseStorage(db_models.PromoSubscription),
            travel_vertical=DatabaseStorage(db_models.TravelVertical),
            email=DatabaseStorage(db_models.Email),
            user_auth_type=DatabaseStorage(db_models.UserAuthType),
            user_auth=DatabaseStorage(db_models.UserAuth),
            user=DatabaseStorage(db_models.User),
            user_promo_subscription=DatabaseStorage(db_models.UserPromoSubscription),
            user_confirm_actor=self.user_confirm_actor,
            session_provider=self.database.session,
            blackbox=self.blackbox,
        )
        self.user_price_change_subscription_actor: UserPriceChangeSubscriptionActor = UserPriceChangeSubscriptionActor(
            travel_vertical=DatabaseStorage(db_models.TravelVertical),
            email=DatabaseStorage(db_models.Email),
            user_auth_type=DatabaseStorage(db_models.UserAuthType),
            user_auth=DatabaseStorage(db_models.UserAuth),
            user=DatabaseStorage(db_models.User),
            price_change_subscription=DatabaseStorage(db_models.PriceChangeSubscription),
            user_price_change_subscription=DatabaseStorage(db_models.UserPriceChangeSubscription),
            user_confirm_actor=self.user_confirm_actor,
            point_key_resolver=self.point_key_resolver,
            session_provider=self.database.session,
            blackbox=self.blackbox,
        )
        self.user_subscription_actor: UserSubscriptionActor = UserSubscriptionActor(
            email=DatabaseStorage(db_models.Email),
            travel_vertical=DatabaseStorage(db_models.TravelVertical),
            user_auth_type=DatabaseStorage(db_models.UserAuthType),
            user_auth=DatabaseStorage(db_models.UserAuth),
            user=DatabaseStorage(db_models.User),
            user_promo_subscription_actor=self.user_promo_subscription_actor,
            user_price_change_subscription_actor=self.user_price_change_subscription_actor,
            session_provider=self.database.session,
            blackbox=self.blackbox,
        )
        self.user_subscription_list_actor: UserSubscriptionListActor = UserSubscriptionListActor(
            email=DatabaseStorage(db_models.Email),
            user_auth_type=DatabaseStorage(db_models.UserAuthType),
            user_auth=DatabaseStorage(db_models.UserAuth),
            user=DatabaseStorage(db_models.User),
            user_promo_subscription_actor=self.user_promo_subscription_actor,
            user_price_change_subscription_actor=self.user_price_change_subscription_actor,
            session_provider=self.database.session,
        )

    async def ping(self) -> PingResponse:
        resolverping = await self.point_key_resolver.ping()
        dbping = self.database.ping()
        info = {
            'application': self.__class__.__name__,
            'config': str(self.config),
            'database': str(dbping),
            'resolver': str(resolverping.info),
        }
        return PingResponse(resolverping.ready, info)
