from aiohttp_apispec import docs, querystring_schema, response_schema
from aiohttp.web import HTTPBadRequest, HTTPServerError
from marshmallow import fields, Schema

from aiohttp.web_response import json_response
from travel.avia.subscriptions.app.api.app import App
from travel.avia.subscriptions.app.api.interactor.user_confirm import (
    InvalidApprovingToken, UserAlreadyApproved
)
from travel.avia.subscriptions.app.lib.sender import SenderError


class UserConfirmGetRequestQueryStringSchema(Schema):
    approving_token = fields.Str(required=True)
    national_version = fields.Str(missing='ru', example='com')


class UserConfirmResponseSchema(Schema):
    confirmed = fields.Bool(required=True)


def get_user_confirm_handler(app: App):
    @docs(
        tags=['user_confirm'],
        summary='Confirm a user from mail',
        description='Confirm a user by his approving token from mail',
        responses={
            200: {'description': 'Ok. User is confirmed'}
        }
    )
    @querystring_schema(UserConfirmGetRequestQueryStringSchema)
    @response_schema(UserConfirmResponseSchema)
    async def get(request):
        data = request['querystring']
        try:
            await app.user_confirm_actor.confirm(
                data['approving_token'],
                data['national_version']
            )
            return json_response(dict(confirmed=True))
        except (UserAlreadyApproved, InvalidApprovingToken) as e:
            return HTTPBadRequest(text='{}'.format(e))
        except SenderError as e:
            return HTTPServerError(text='{}'.format(e))

    return get
